(ns blueprint.client.interceptors
  (:require [aleph.http :as http]
            [exoscale.interceptor :as ix]
            [exoscale.ex :as ex]
            [exoscale.coax :as cx]
            [clojure.tools.logging :as log]
            [muuntaja.core :as m]
            [clojure.spec.alpha :as s]
            [clojure.tools.logging :as log]
            [spec-tools.core :as st]
            [manifold.deferred :as d]))
;;
;; Private
;;
(defn- decodeable?
  "Indicates if the response can be decoded."
  [response]
  (some? (get-in response [:headers "content-type"])))

(defn- conform-body
  "Tries to conform the body according to the command's output spec.
  If decoding was unsuccessful, a `:invalid? true` keyval is added to the response, and the original
  body is left as is."
  [multispec response]
  (let [decoded  (st/decode multispec response st/json-transformer)
        invalid? (s/invalid? decoded)]
    (if invalid?
      (do
        ;; this will complain alot for undocumented return codes
        ;; such as 4xx or 5xx, so we keep it debug
        (log/debugf "Response does not conform: %s"
          (with-out-str
            (st/explain multispec response st/json-transformer)))
        (assoc response :invalid? true))
      decoded)))

(defn- decode-body
  "Decodes the http request body according to content type headers."
  [{:keys [headers] :as response}]
  ;; unfortunately, aleph http returns ring headers (lowercased)
  ;; but muuntaja expects Pascal-Case headers
  ;; so a hack is needed to ensure the m/decode-body-response finds the correct content-type
  ;; the alternative is to call m/decode-response-body with content-type and charset args
  ;; (m/decode response "application/edn" "utf-8"
  (let [norm-headers (assoc-in response [:headers "Content-Type"] (get headers "content-type"))]
    (->> norm-headers
      (m/decode-response-body)
      (assoc response :body))))

(defn- try-parse-body
  "Tries to manually parse the body. Does a best-effort to parse the body.
  If the body could not be parsed, returns the raw stream."
  [{:keys [request response] :as ctx}]
  ;; m/decode-response-body, via jackson, barfs if an empty input stream is decoded
  ;; https://github.com/FasterXML/jackson-databind/issues/907
  ;; it's hard to reliably check for an empty stream
  ;; (.available (:body response)) returns zero sometimes (when the stream is not yet :aleph/complete ?)
  ;; so we can't reliably set the body to nil when the response stream says available=0
  (try
    (let [format  (or (get-in request [:headers "Accept"])
                    (m/default-format m/instance))
          charset (m/default-charset m/instance)
          body    (:body response)
          decoded (m/decode m/instance format body charset)]
      (assoc-in ctx [:response :body] decoded))
    (catch Throwable e
      (log/error e "Could not parse response body")
      ctx)))


(defn- status-code->ex-type
  "Try to resolve a status code to a type. A status code may be mapped to several types,
  so we return the most 'generic'."
  [code]
  (get {400 ::ex/incorrect
        404 ::ex/not-found
        403 ::ex/forbidden
        405 ::ex/unsupported
        409 ::ex/conflict
        504 ::ex/unavailable
        503 ::ex/busy
        500 ::ex/default} code
    ::ex/fault))

(defn- decode-error
  "Tries to parse the HTTP error response, so that the response contains a parsed body instead of a stream."
  [{:keys [request] :as ctx} e]
  (if-not (:decode-on-error request)
    (d/error-deferred e)
    ;; else
    (try
      (let [data    (ex-data e)
            status  (:status data)
            msg     (ex-message e)
            decoded (m/decode-response-body data)
            ;; this has a problem, if the content-type is json
            ;; some *values* won't be reconverted as keywords
            ;; so we use coax to try and coerce the data
            coerced (cx/coerce ::ex/ex-map decoded)
            ex-map? (s/valid? ::ex/ex-map coerced)]

        (if ex-map?
          ;; no need to assoc :body because m/decode-response-body will consume the input stream
          ;; it would appear as an empty body
          (d/error-deferred (ex/map->ex-info (assoc-in coerced [:exoscale.ex/data :status] status) {:derive? true}))
          (d/error-deferred
            (ex/ex-info msg  (status-code->ex-type status) (assoc decoded :status status)))))#_
            (ex-info msg (assoc decoded :status status :type (status-code->ex-type status)))

      (catch Exception ex
        (log/error ex "Could not parse error response body")
        ;; return original exception
        (d/error-deferred e)))))

(defn- http-call [{:keys [request] :as ctx}]
  (-> (d/chain (http/request request)
               #(assoc ctx :response %))
    (d/catch #(decode-error ctx %))))

(def execute-request
  {:name  ::execute-request
   :enter #'http-call})

(def lens-response
  {:name  ::lens-response
   :enter (ix/lens identity [:response])})

(def get-response
  {:name  ::get-response
   :leave (ix/in identity [:response])})

(def parse
  {:name  ::parse-body
   :enter (-> try-parse-body
            (ix/when #(not (decodeable? (:response %)))))})

;; decoder requires a response-multispec
(s/def ::response-spec s/spec?)
(s/def ::decoder-config (s/keys :req-un [::response-spec]))
(def decode
  {:name    ::decode
   :spec    ::decoder-config
   :builder (fn build-decoder [interceptor {:keys [response-spec] :as config}]
              (let [conform-body-fn (partial conform-body response-spec)]
                (-> interceptor
                  (assoc :enter (-> (comp conform-body-fn decode-body)
                                  (ix/lens [:response])
                                  (ix/when #(decodeable? (:response %))))))))})
