(ns blueprint.client
  (:require [blueprint.client.url :as url]
            [blueprint.core :as core]
            [blueprint.registry :as reg]
            [blueprint.interceptor :as bpi]
            [blueprint.client.interceptors :as interceptors]
            [aleph.http :as http]
            [muuntaja.core :as m]
            [exoscale.interceptor.manifold :as ixm]
            [blueprint.client.spec-gen :as bsg]
            [clojure.spec.alpha :as s]
            [exoscale.ex :as ex]))

(def default-interceptors
  [interceptors/execute-request
   interceptors/lens-response
   interceptors/decode
   interceptors/parse
   interceptors/get-response])

(defn- build-chain
  "Build an interceptor chain in three phases:

  - Start from the default chain
  - Process additional interceptors provided in `::additional`
  - Removes interceptors by name based on those provided in `::disabled`
  - Call the interceptor build step if any with the provided config"
  [config]
  (ex/assert-spec-valid ::interceptors config)
  (bpi/build-chain default-interceptors config))

(def build-with bpi/build-with)

;;
;; Http Client
;;

(defn- base-url
  "Gets the base url for the defined api."
  [{:keys [servers] :as parsed-api}]
  (if-let [prefix (get-in parsed-api [:blueprint.options :path-prefix])]
    (str (:url (first servers)) prefix)
    (:url (first servers))))

(defrecord Client [parsed-api command-defs servers connection-pool])

(defn make-client
  "Create a blueprint client from a blueprint api definition."
  ([api-def] (make-client api-def {}))
  ([api-def {:keys [ssl-context] :as default-options}]
   (ex/assert-spec-valid ::api-definition api-def)
   (let [parsed-api   (core/parse api-def)
         command-defs (bsg/apidef->commands-map parsed-api)
         connection-pool (if ssl-context
                           (http/connection-pool
                            {:connection-options {:ssl-context ssl-context}})
                           http/default-connection-pool)]
     (->Client parsed-api command-defs (:servers api-def) connection-pool))))

(def default-options
  {:pool-timeout       10e3
   :connection-timeout 10e3
   :request-timeout    10e3
   :read-timeout       10e3})

(defn invoke
  "Invoke a `command` on the remote blueprint-based api.
  Clients are created with `(blueprint.client/make-client api-def)`.
  Supported options in `opts`:

  | key                   | description |
  | ----------------------|-------------|
  | `:input`              | The request input to be sent
  | `:server`             | The remote server endpoint (eg: \"http://localhost:8080\"); default is first entry from api definition
  | `:headers`            | A map of additional headers to be sent
  | `:options`            | A map of additional options to add to the request
  | `:as`                 | The format for content negotation; accepts `edn` or `json`; default: `json`
  | `:interceptors`       | The interceptor configuration to apply
  | `:throw-exceptions`   | Control throwing of exceptions in case of errors
  "
  [{:keys [parsed-api command-defs] :as client}
   command
   {:keys [input
           server
           headers
           options
           as
           throw-exceptions
           interceptors]
    :or   {input   nil
           server  (base-url parsed-api)
           headers {}
           as      :json
           throw-exceptions true
           interceptors {}}
    :as   opts}]

  (ex/assert-spec-valid ::client client)
  (ex/assert-spec-valid ::command command)
  (ex/assert-spec-valid ::options opts)
  (ex/assert-spec-valid ::interceptors interceptors)
  ;;ensure command is valid
  (ex/assert-spec-valid (into #{} (keys command-defs)) command {:message "Invalid command"})

  (let [command-def              (get command-defs command)
        {::reg/keys [commands]} parsed-api
        {:keys [input-spec
                path-spec
                params-spec
                response-spec
                input?
                params?
                pathelems?]}     command-def
        command-map              (get commands command)
        request-map              (url/cmd->request command-map input)
        format                   (cond
                                   (= as :edn) "application/edn"
                                   :else       "application/json")]

    ;; validate command specifics
    (if input?     (ex/assert-spec-valid input-spec (:body request-map)) {:message "Invalid input"})
    (if pathelems? (ex/assert-spec-valid path-spec input {:message "Invalid input path params"}))
    (if params?    (ex/assert-spec-valid params-spec (:query-params request-map) {:message "Invalid input params"}))

    (let [target-url        (str server (:url request-map))
          headers           (cond-> (assoc headers "Accept" format)
                              input? (assoc "Content-Type" format))
          base-req          (-> request-map
                                (merge default-options options)
                                (assoc :url target-url
                                       :headers headers
                                       :throw-exceptions throw-exceptions
                                       :pool (:connection-pool client)))
          http-req          (cond-> base-req
                              input? (assoc :body (m/encode format input))
                              params? (assoc :query-params (:query-params request-map)))
          interceptor-chain (build-chain (merge interceptors command-def))]

      (ixm/execute {:request http-req} interceptor-chain))))


;;;;
;; Specs
;;;;

;; api-definition


(s/def ::api-definition ::core/definition)

;; client
(s/def ::client #(instance? Client %))

;; invoke arguments
(s/def ::command keyword?)

;; config
(s/def ::input map?)
(s/def ::server string?)
(s/def ::as #{:json :edn})
(s/def ::throw-exceptions boolean?)
(s/def ::user-interceptors map?)
(s/def ::interceptors (s/keys :opt [::bpi/additional ::bpi/disabled]))
(s/def ::options (s/nilable (s/keys :opt-un [::input ::server ::headers
                                             ::as ::throw-exceptions
                                             ::interceptors])))
