(ns blueprint.client
  (:require [blueprint.client.url :as url]
            [blueprint.client.validation :as validation]
            [blueprint.core :as core]
            [aleph.http :as http]
            [manifold.deferred :as d]
            [muuntaja.core :as m]
            [blueprint.client.spec-gen :as bsg]
            [spec-tools.core :as st]
            [clojure.spec.alpha :as s])
  (:import [java.io InputStream]))

;;
;; Private
;;
(defn- decodeable?
  "Indicates if the response can be decoded."
  [response]
  (some? (get-in response [:headers "Content-Type"])))

(defn- conform-body
  "Tries to conform the body according to the command's output spec.
  If decoding was unsuccessful, a `:invalid? true` keyval is added to the response, and the original
  body is left as is."
  [command multispec {body :body :as res}]
  (let [decoded  (st/decode multispec (assoc body :handler command) st/json-transformer)
        invalid? (s/invalid? decoded)]
    (merge res
      (if invalid? {:invalid? true}
                   {:body decoded}))))

(defn- decode-body
  "Decodes the http request body according to content type headers."
  [{:keys [body] :as response}]
  ;;muuntaja throws if body is nil and there is no content-type header
  (if (and (not (decodeable? response))
           (= (.available ^InputStream body) 0))
    (assoc response :body nil)
    ;;else
    (->> (m/decode-response-body response)
         (assoc response :body))))

;;
;; Http Client
;;

(defrecord Client [parsed-api command-defs servers])

(defn make-client
  "Create a blueprint client from a blueprint api definition."
  [api-def]
  (let [parsed-api   (core/parse api-def)
        command-defs (bsg/apidef->commands-map parsed-api)]
    (->Client parsed-api command-defs (:servers api-def))))


(defn invoke
  "Invoke a `command` on the remote blueprint-based api.
  Clients are created with `(blueprint.client/make-client api-def)`.
  Supported options in `opts`:

  | key           | description |
  | --------------|-------------|
  | `:input`      | The request input to be sent
  | `:server`     | The remote server endpoint (eg: \"http://localhost:8080\"); default is first entry from api definition
  | `:headers`    | A map of headers to be sent
  | `:as`         | The format for content negotation; accepts `edn` or `json`; default: `json`
  "
  [{:keys [parsed-api command-defs servers] :as client}
   command
   {:keys [input
           server
           headers
           as]
    :or   {input   nil
           server  (:url (first servers))
           headers {}
           as      :json}
    :as   opts}]

  ;;ensure command is valid
  (validation/validate-command (into #{} (keys command-defs)) command)

  (let [command-def    (get command-defs command)
        {:keys [commands specs]} parsed-api
        {:keys [input-spec
                path-spec
                params-spec
                input?
                params?
                pathelems?]} command-def
        command-map    (get commands command)
        request-map    (url/cmd->request command-map input)
        resp-multispec (:handler specs)
        format         (cond
                         (= as :edn) "application/edn"
                         :else       "application/json")]

    (if input? (validation/validate-input input-spec (:body request-map)))
    (if pathelems? (validation/validate-path path-spec input))
    (if params? (validation/validate-params params-spec (:query-params request-map)))

    (let [http-req     (merge request-map
                              {:url     (str server (:url request-map))
                               :headers (cond-> (merge headers {"Accept" format})
                                          input? (merge {"Content-Type" format}))}
                              (if input? {:body (m/encode format input)})
                              (if params? {:query-params (:query-params request-map)}))

          conform-body (partial conform-body command resp-multispec)]

      (d/chain (http/request http-req)
               decode-body
               conform-body))))
