(ns blueprint.client
  (:require [blueprint.client.url :as url]
            [blueprint.client.validation :as validation]
            [blueprint.core :as core]
            [aleph.http :as http]
            [manifold.deferred :as d]
            [cheshire.core :as json]
            [blueprint.client.spec-gen :as bsg]
            [spec-tools.core :as st]
            [clojure.spec.alpha :as s]))

;;
;; Http Client 
;;

(defrecord Client [parsed-api command-defs servers])

(defn make-client
  "Create a blueprint client from a blueprint api definition."
  [api-def]
  (let [parsed-api   (core/parse api-def)
        command-defs (bsg/apidef->commands-map parsed-api)]
    (->Client parsed-api command-defs (:servers api-def))))


(defn- decode-body
  "Tries to decode the body according to the command's output spec.
  If decoding was unsuccessful, a `invalid? true` keyval is added to the response, and the original
  body is left as-is."
  [command multispec {body :body :as res}]
  (let [decoded  (st/decode multispec (assoc body :handler command) st/json-transformer)
        invalid? (s/invalid? decoded)]
    (merge res
      (if invalid? {:invalid? true}
                   {:body decoded}))))

(defn invoke
  "Invoke a command on the blueprint client. Assumes json for content negotiation."
  [{:keys [parsed-api command-defs servers] :as client}
   command
   {:keys [input
           server
           headers]
    :or   {input   nil
           server  (:url (first servers))
           headers (merge {"Accept" "application/json"}
                     (when input {"Content-Type" "application/json"}))}
    :as   opts}]

  ;;ensure command is valid
  (validation/validate-command (into #{} (keys command-defs)) command)

  (let [command-def    (get command-defs command)
        {:keys [commands specs]} parsed-api
        {:keys [input-spec
                path-spec
                params-spec
                input?
                params?
                pathelems?]} command-def
        command-map    (get commands command)
        request-map    (url/cmd->request command-map input)
        resp-multispec (:handler specs)]

    (if input? (validation/validate-input input-spec (:body request-map)))
    (if pathelems? (validation/validate-path path-spec input))
    (if params? (validation/validate-params params-spec (:query-params request-map)))

    (let [http-req (merge request-map
                     {:url     (str server (:url request-map))
                      :headers headers
                      :as      :json}
                     (if input? {:body (json/generate-string input)})
                     (if params? {:query-params (:query-params request-map)}))]

      (d/chain (http/request http-req)
               (partial decode-body command resp-multispec)))))
