(ns blueprint.client.spec-gen
  (:require [blueprint.spec-gen :as sg]
            [spec-tools.data-spec :as ds]
            [clojure.string :as s]))

;;
;; Spec gen from api def
;;

(defn- pathelems->spec
  "Converts a seq of path elems to a spec. Path elems are taken from a blueprint def
  and have the following shape:

```clojure
    ([:string \"/job\"]
     [:arg {:name :id, :def {:type :pred, :pred uuid?}}])
```"
  [cmdname elems]
  (let [dynelems (filter (fn [[t _]] (= :arg t)) elems)
        dynelems (map second dynelems)
        names    (map (comp name :name) dynelems)]

    (when (seq dynelems)
      (ds/spec {:name (sg/genq (str cmdname "-" (s/join "-" names)))
                :spec (reduce
                        (fn [acc {name :name def :def}]
                          (assoc acc name (sg/compile-spec def)))
                        {}
                        dynelems)}))))

(defn- params-map->spec
  "Converts a map of `{key specdef}` to a spec.
  All keys are optional. The map has the following shape:

```clojure
    {:company {:type :pred, :pred string?}
     :page    {:type :pred, :pred pos-int?}}
```"
  [cmdname m]
  (when (seq m)
    (ds/spec {:name (sg/genq (str cmdname "-params"))
              :spec (reduce-kv
                      (fn [m k v]
                        ;; every param is optional
                        (assoc m (ds/opt k) (sg/compile-spec v)))
                      {}
                      m)})))

(defrecord CommandDef [name input-spec path-spec params-spec response-spec input? params? pathelems?])

(defn apidef->commands-map
  "Generates a map of command definitions from a parsed api definition"
  [parsed-api]
  (let [{:keys [commands specs output-specs]} parsed-api
        command-keys (keys commands)]
    (apply merge
      (for [command command-keys
            :let [input-spec     (get specs command)
                  command-map    (get commands command)
                  path-elems     (get-in command-map [:path :elems])
                  params-map     (get command-map :params)

                  pathelems-spec (pathelems->spec command path-elems)
                  params-spec    (params-map->spec command params-map)
                  response-mspec (get output-specs command)]]

        {command (->CommandDef command
                   input-spec
                   pathelems-spec
                   params-spec
                   response-mspec
                   (contains? (get commands command) :input)
                   (some? params-spec)
                   (some? pathelems-spec))}))))
