(ns blueprint.assets
  (:require [clojure.string :as str]
            [ring.util.response :as response]
            [ring.util.mime-type :as mime-type]
            [clojure.spec.alpha :as s]))

(defn- is-asset-request?
  "Indicates if the incoming request is an asset request.
  A request is an asset request if its uri starts with the defined `asset-path`"
  [method uri assets-path]
  (and
    (#{:get :head} method)
    (str/starts-with? uri assets-path)))

(defn- find-asset
  ([path]
   (find-asset path nil))
  ([path root]
   ;(response/resource-response path {:root root})
   (response/file-response path {:root root})))

(defn- serve-asset
  "Serves an asset for the given `path` looking at possible `roots`. Returns `nil` if not found"
  [path roots]
  (some #(find-asset path %) roots))

(defn generate-ring-handler
  "Generates a ring handler for asset managing. `api-def` should have an `assets` key:

  ```
  :router
  {:assets [\"/public\" [\"root1\" ... \"rootN\"]}
  ```

  The assets will be served from the *classpath*, with the optional `root`.
  All roots will be sequentially tested until one of them returns an asset.
  "
  [{:keys [assets] :as api-def}]
  (if (empty? assets)
    (constantly nil)

    (let [{:keys [path roots]} assets]
      (fn [{:keys [request-method uri]}]
        (if (is-asset-request? request-method uri path)
          (some->
           (serve-asset (subs uri (count path)) roots)
           (response/content-type (mime-type/ext-mime-type uri))))))))
