(ns leiningen.gzip-resources
  (:require [clojure.java.io :as io]
            [clojure.string :as str])
  (:import (java.io File)
           (java.util.zip GZIPOutputStream)
           (org.apache.commons.io IOUtils)))

(defn length->str [length]
  (let [[length unit] (loop [length (double length)
                             [unit & units] ["bytes" "kB" "MB" "GB" "TB"]]
                        (if (or (<= length 1024.0) (empty? units))
                          [length unit]
                          (recur (/ length 1024.0) units)))]
    (format "%.1f %s" length unit)))

(defn relative-path [^File base ^File file]
  (-> base
      (.toURI)
      (.relativize (-> file .toURI))
      (.getPath)))

(defn gzip-resource [^File target-dir [^File resource-dir ^File resource-file]]
  (let [input-length  (.length resource-file)
        resource-name (relative-path resource-dir resource-file)
        output-file   (io/file target-dir (str "resources/" resource-name ".gz"))]
    (print (str "Gzipping " resource-name ": " (length->str input-length)))
    (-> output-file .getParentFile .mkdirs)
    (with-open [in  (-> resource-file
                        (io/input-stream))
                out (-> output-file
                        (io/output-stream)
                        (GZIPOutputStream.))]
      (io/copy in out))
    (let [output-length (.length output-file)
          ratio         (/ (- input-length output-length) (double input-length))
          worth-it?     (> ratio 0.1)]
      (println (format " -> %s (%+.1f%%%s)"
                       (length->str output-length)
                       (* ratio -100.0)
                       (if-not worth-it? " gzip skipped" "")))
      (when-not worth-it?
        (.delete output-file))
      (merge {:resource        resource-name
              :resource-length input-length}
             (when worth-it?
               {:resource-gz        (str resource-name ".gz")
                :resource-gz-length output-length})))))

(defn regular-file? [^File file]
  (.isFile file))

(defn public? [^File file]
  (-> file (.getPath) (str/index-of "/public/") (some?)))

(defn find-resources [resource-dir]
  (->> (file-seq resource-dir)
       (filter regular-file?)
       (filter public?)
       (map (fn [resource-file]
              [resource-dir resource-file]))))

(defn gzip-resources
  "Find and gzip resources. Generates an gzipped file of every resource. Generated
  file name is same as the resource name with \".gz\" appended to it.

  Supported options:
    :target-path     - Target path, defaults to project :target
    :resource-paths  - Seq of resource paths, defaults to project :resource-paths
    :manifest-path   - If set, generates an edn file with resource information

  Options are read under key :gzip-resources in project.clj"
  [project & args]
  (let [opts          (merge {:target-path    (-> project :target-path)
                              :resource-paths (-> project :resource-paths)
                              :manifest-path  nil}
                             (-> project :gzip-resources)
                             (loop [acc {}
                                    [k v & args] args]
                               (if k
                                 (recur (assoc acc (keyword k) v)
                                        args)
                                 acc)))
        target        (-> (or (some-> opts :target-path)
                              (some-> project :target-path))
                          (io/file))
        resource-dirs (->> (or (some-> opts :resource-paths)
                               (some-> project :resource-paths))
                           (set)
                           (map io/file))
        manifest      (->> resource-dirs
                           (mapcat find-resources)
                           (doall) ; ensure that the disk is scanned before we start to add files to disk
                           (keep (partial gzip-resource target))
                           (into []))]
    (when-let [manifest-path (-> opts :manifest)]
      (println "Writing resource manifest to" manifest-path)
      (with-open [out (-> (io/file target manifest-path)
                          (io/writer))]
        (.write out (pr-str manifest))))
    nil))
