;; Copyright (C) 2013, Jozef Wagner. All rights reserved.
;;
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0
;; (http://opensource.org/licenses/eclipse-1.0.php) which can be
;; found in the file epl-v10.html at the root of this distribution.
;;
;; By using this software in any fashion, you are agreeing to be bound
;; by the terms of this license.
;;
;; You must not remove this notice, or any other, from this software.

(ns eu.us.settings.core
  "Application-wide settings."
  (:refer-clojure :exclude [get])
  (:require [clojure.edn :as edn]
            [clojure.java.io :as io])
  (:import [java.io PushbackReader])
  (:gen-class
   :name eu.us.settings.Settings
   :methods [^:static [get [String] Object]
             ^:static [get [String String] Object]
             ^:static [get [String String String] Object]
             ^:static [get [String String String String] Object]
             ^:static [get [String String String String String]
                       Object]
             ^:static [get [String String String String String String]
                       Object]
             ^:static [get [String String String String String String
                            String] Object]]))

;;;; Implementation details

(def settings-url "settings.edn")

(defn- read-settings
  "Read settings map from a specified resource."
  ([]
     (read-settings settings-url))
  ([url]
     (-> url
         io/resource
         io/reader
         PushbackReader.
         edn/read)))

(defn- string-map
  "Transforms input map so all keys are string."
  [m]
  (if-not (map? m)
    m
    (let [k (map #(if (keyword? %) (name %) (str %)) (keys m))
          v (map string-map (vals m))]
      (when-not (== (count (set k)) (count (keys k)))
        (throw (Exception. "Duplicate key. Check that no two keys at given level produce same string representation.")))
      (apply hash-map (interleave k v)))))

(defn -get
  "Get function for interop."
  [& ks]
  (let [s (string-map (read-settings))]
    (clojure.core/get-in s ks)))

;;;; Public API

(defn get
  "Returns value for a given keys.
  Returns nil if no value is set for given keys"
  [& ks]
  (let [s (read-settings)]
    (clojure.core/get-in s ks)))

;;;; Usage

;;; for usage, see settings-example project in example directory
