(ns herark.smi.smiv1
  "SMIv1 types defined as Prismatic Schema schemas. They are present in SNMPv1 traps,
  both in the PDU definition and the include _variable bindings_ (`VarBinds`). They are
  slightly different from the SMIv2 types that are present in other SNMP versions.

  They can be classified into:

  * Types for SMI Values (SMIv1Value): OID, Int, OctetString, ... They are mapped to
      Cloure tag-value pairs.
  * VarBinds (SMIv1VarBind): pairs of SMI Values being the first an OID
  * SNMPv1 PDU (V1TrapPDU): Includes the SNMPv1 Trap PDU fields as defined by the standards.
  * SNMPv1 Message (V1TrapMessage): Includes the source address, the community and a PDU.

  ### Interesting functions

  This namespace has a number of vars generated by Schema. Letting them aside,
  the main functions to have a look at are `make-v1-trap-message` and `make-v1-trap-pdu`.

  ### Misc vars

  This namespace includes constants for generic traps."

  (:require [schema.core :as s]
            [herark.smi.misc :refer :all]))

;; Acording to RFC1155
;; https://www.ietf.org/rfc/rfc1155.txt

(defn- oid-value?
  "Is `x` a valid oid according to the SMI rules?"
  [x]
  (and
    (sequential? x)
    (not (empty? x))
    (#{0 1 2} (first x))
    (every? (fn [y]
              (and (integer? y) (>= y 0))) x)))

(s/defschema OID
  "Schema for SMI OIDs."
  (tag-value-pair ::oid oid-value?))

(defn- int-value?
  "Is `x` a valid Integer according to the SMI rules?"
  [x]
  (and
    (integer? x)))

(s/defschema Int
  "Schema for SMI Integers."
  (tag-value-pair ::int int-value?))

(defn- counter-value?
  "Is `x` a valid Counter according to the SMI rules?"
  [x]
  (and
    (integer? x)
    (>= x 0)
    (<= x TWO-32-MINUS-1)))

(s/defschema Counter
  "Schema for SMI Counter"
  (tag-value-pair ::counter counter-value?))

(defn- gauge-value?
  "Is `x` a valid Gauge according to the SMI rules?"
  [x]
  (and
    (integer? x)
    (>= x 0)
    (<= x TWO-32-MINUS-1)))

(s/defschema Gauge
  "Schema for SMI Gauge"
  (tag-value-pair ::gauge gauge-value?))

(defn- time-ticks-value?
  "Is `x` a valid TimeTicks value according to the SMI rules?"
  [x]
  (and
    (integer? x)
    (>= x 0)))

(s/defschema TimeTicks
  "Schema for SMI TickeTicks"
  (tag-value-pair ::time-ticks time-ticks-value?))

(defn- octet-string-value?
  "Is `x` a valid OctetString according to the SMI rules?"
  [x]
  (and
    (sequential? x)
    (every? byte-value? x)))

(s/defschema OctetString
  "Schema for SMI OctetString."
  (tag-value-pair ::octet-string octet-string-value?))

(defn- opaque-value?
  "Is `x` a valid Opaque value according to the SMI rules?"
  [x]
  (octet-string-value? x))

(s/defschema Opaque
  "Schema for SMI Opaque."
  (tag-value-pair ::opaque opaque-value?))

(defn- ip-address-value?
  "Is `x` a valid IPAddress value according to the SMI rules?"
  [x]
  (and
    (string? x)
    (re-matches IP-RE x)))

(s/defschema IPAddress
  "Schema for SMI IPAdress."
  (tag-value-pair ::ip-address ip-address-value?))

(s/defschema SMIv1Value
  "Schema for SMI Values."
  (s/either OID
            Int
            Gauge
            Counter
            TimeTicks
            OctetString
            Opaque
            IPAddress))

(s/defschema SMIv1VarBind
  "Schema for SMI Variable Bindings."
  (s/pair OID "oid" SMIv1Value "variable"))

;FIXME determine if we need NetworkAddress
(s/defrecord V1TrapPDU [enterprise :- OID
                        source-address :- IPAddress
                        generic-trap-type :- Int
                        specific-trap-type :- Int
                        timestamp :- TimeTicks
                        varbinds :- [SMIv1VarBind]])

(s/defn make-v1-trap-pdu :- V1TrapPDU
  "Creates an SNMPv1 trap PDU after receiving tagged values as parameters. It does
  _little_ beyond providing a uniform interface and type checking:

  * `source-address`:       `[:smiv1/ip-address \"192.168.0.1\"]`
  * `generic-trap-type`:    `[:smiv1/int 3]`
  * `enterprise`:           `[:smiv1/octet-string [1 3 6 1 4 1 XXX]]`
  * `timestamp`:            `[:smiv1/time-ticks 132132]`
  * `varbinds`:             `[[[:smiv1/oid [1 3 ...] [:smiv1/int 1984]]]`."
  [source-address :- IPAddress
   generic-trap-type :- Int
   enterprise :- OID
   specific-trap-type :- Int
   timestamp :- TimeTicks
   varbinds :- [SMIv1VarBind]]
  (->V1TrapPDU enterprise source-address generic-trap-type specific-trap-type timestamp varbinds))

;; SNMP v1 Trap Message
(s/defrecord ^{:doc "SNMP v1 Trap Message. It contains the PDU as well as other data."} V1TrapMessage
  [version :- (s/eq :v1)
   source-address :- IPAddress
   community :- OctetString
   pdu :- V1TrapPDU])

(s/defn make-v1-trap-message :- V1TrapMessage
  "Creates an SNMPv1 message receiving tagged values as parameters.

  * `source-address`:    `[:smiv1/ip-address \"192.168.0.1\"]`
  * `community`:         `[:smiv1/octet-string [65 66 67]]`
  * `pdu`:               see `make-v1-trap-pdu`

  The source-address fields in the message and in the PDU may be
  different."
  [source-address :- IPAddress
   community :- OctetString
   pdu :- V1TrapPDU]
  (->V1TrapMessage :v1 source-address community pdu))

(def
  ^{:const true :doc "Integer value for the generic trap AUTHENTICATION FAILURE"}
  AUTHENTICATION-FAILURE-TRAP-IDX 4)
(def
  ^{:const true :doc "Integer value for the generic trap COLD START"}
  COLD-START-TRAP-IDX 0)
(def
  ^{:const true :doc "Integer value for the flag for traps that are enterprise specific"}
  ENTERPRISE-SPECIFIC-TRAP-IDX 6)
(def
  ^{:const true :doc "Integer value for the generic trap LINK DOWN"}
  LINK-DOWN-TRAP-IDX 2)
(def
  ^{:const true :doc "Integer value for the generic trap LINK UP"}
  LINK-UP-TRAP-IDX 3)
(def
  ^{:const true :doc "Integer value for the generic trap WARM START"}
  WARM-START-TRAP-IDX 1)

(def
  ^{:const true :doc "OID prefix for generic traps."}
  GENERIC-TRAP-OID-PREFIX
  [1 3 6 1 6 3 1 1 5])
