(ns formform.calc.specs
  (:require
   [formform.calc.core :as core]
   [formform.utils :as utils]
   [clojure.spec.alpha :as s]
   [clojure.spec.gen.alpha :as gen])
  #?(:cljs (:require-macros
            [formform.calc.specs :refer [spec--dna-args spec--dna-seq-args]])))

;;-------------------------------------------------------------------------
;; constant

(s/def ::val-hole #(= % :_))

(s/def ::const
  (s/with-gen
    #(case % (:n :u :i :m) true false)
    #(gen/elements [:n :u :i :m])))
(s/def ::const_
  (s/with-gen
    (s/or :const    ::const
          :val-hole ::val-hole)
    #(gen/elements [:n :u :i :m :_])))

(s/def ::consts (s/every ::const
                         :kind sequential?
                         :min-count 1))
(s/def ::consts_ (s/every ::const_
                          :kind sequential?
                          :min-count 1))

(s/def ::const-int (s/int-in 0 4))
(s/def ::const_-int (s/int-in -1 4))

(s/def ::const-char #{\N \U \I \M, \n \u \i \m, \0 \1 \2 \3})
(s/def ::const_-char #{\N \U \I \M, \n \u \i \m, \0 \1 \2 \3  \- \_})

(s/def :rand/seed (s/nilable nat-int?))

(s/def ::const-weights ;; random weights
  (s/or :sequence (s/coll-of number? :kind sequential? :count 4)
        :map (s/map-of #{:n :u :i :m} number?)
        :ratio (s/and number? #(<= 0.0 % 1.0))))

(s/def ::sort-code
  (s/with-gen
    (s/and vector?
           #(== 4 (count %))
           #(= #{:n :u :i :m} (set %)))
    #(gen/shuffle [:n :u :i :m])))

;;-------------------------------------------------------------------------
;; formDNA

(s/def ::dna-length
  (s/with-gen
    #(some? (core/dna-length->dim %))
    #(gen/elements (take 12 core/dna-lengths)))) 

(s/def ::dna-dimension nat-int?)

;; ? necessary
(s/def ::dna-count #(some? (core/dna-dimension %)))


(s/def ::dna
  (s/and (s/coll-of ::const
                    :kind sequential?
                    :min-count 1)
         (comp (partial s/valid? ::dna-length)
               count)))

(s/def ::dna_ ; allows holes
  (s/and (s/coll-of ::const_
                    :kind sequential?
                    :min-count 1)
         (comp (partial s/valid? ::dna-length)
               count)))

(s/def ::dna-seq
  (s/and sequential? #(<= (count (set %)) 4)
         ::dna-count))

(s/def ::dna-seq_ ; allows holes
  (s/and sequential? #(<= (count (set %)) 5)
         ::dna-count))


(s/def ::dna-seq-elem
  (s/or :const ::const
        :char  ::const-char
        :int   ::const-int))

(s/def ::dna-seq-elem-tree ;; required by `core/make-dna`
  (s/coll-of (s/or :leaf   ::dna-seq-elem
                   :branch ::dna-seq-elem-tree)
             :kind (complement map?)))


;;-------------------------------------------------------------------------
;; formDNA perspective

(s/def ::permutation-order
  (s/coll-of nat-int?
             :distinct true
             :kind sequential?))

(s/def ::dna-perspective
  (s/cat :perm-order ::permutation-order
         :dna        ::dna_))

(s/def ::dna-perspective-group
  (s/map-of ::permutation-order ::dna_))


;;-------------------------------------------------------------------------
;; vpoint

(s/def ::vpoint
  (s/every ::const_
           :kind sequential?))

;;-------------------------------------------------------------------------
;; vspace

;; ? should vector be enforced
;; - pro: random-access makes more sense for coordinates
;; - pro: formDNA is already a vector for good reasons
;; - con: generated by cartesian product, which returns a lazy seq
;; - con: might be better for large spaces
(s/def ::vspace
  (s/and (s/coll-of ::vpoint
                    :min-count 1
                    :distinct true)
         (fn [vspc] (let [vs-dim (core/dna-dimension (seq vspc))
                         vp-dim (count (first vspc))]
                     (and (some? vs-dim)
                          (== vs-dim vp-dim)
                          (every? #(== vp-dim (count %)) vspc))))))
;; ? add spec ordered-vspace

;;-------------------------------------------------------------------------
;; vdict

(s/def ::vdict
  (s/and map?
         #(s/valid? ::vspace  (keys %))
         #(s/valid? ::dna-seq (vals %))))

;;-------------------------------------------------------------------------
;; vmap

(s/def ::vmap
  (s/or :vmap (s/map-of ::const
                        ::vmap
                        :count 4)
        :res ::const))

(s/def ::vmap-perspective
  (s/cat :perm-order ::permutation-order
         :vmap       ::vmap))

(s/def ::vmap-perspective-group
  (s/map-of ::permutation-order ::vmap))

;;-------------------------------------------------------------------------
;; function specs (impl)


;; note: dropped the "4r" prefix requirement
(let [digits #{\0 \1 \2 \3}]
  (s/def ::quaternary-str (s/and string? (partial every? digits))))

(s/fdef core/consts->quaternary
  :args (s/alt :ar1 (s/cat :consts ::consts)
               :ar2 (s/cat :consts ::consts
                           :rtl? boolean?))
  :ret  ::quaternary-str)


(defmacro spec--dna-seq-args [spec]
  `(s/alt :ar1 (s/cat :dna-seq ~(if (nil? spec)
                                  `::dna-seq_
                                  `(s/and ::dna-seq_ ~spec)))
          :ar2 (s/cat :sort-code ::sort-code
                      :dna-seq ~(if (nil? spec)
                                  `::dna-seq_
                                  `(s/and ::dna-seq_ ~spec)))))

(defmacro spec--dna-args []
  `(s/alt :ar1 (s/cat :dna       ::dna_)
          :ar2 (s/cat :sort-code ::sort-code
                      :dna       ::dna_)))

(s/fdef core/prod=dna-seq->dna
  :args (s/fspec :args (s/cat :sort-code ::sort-code
                              :x         any?)
                 :ret  ::const)
  :ret  (s/fspec :args (spec--dna-seq-args nil)
                 :ret  ::dna_))

(s/fdef core/prod=dna->dna-seq
  :args (s/fspec :args (s/cat :sort-code ::sort-code
                              :const     ::const)
                 :ret  any?)
  :ret  (s/fspec :args (spec--dna-args)
                 :ret  ::dna-seq_))


(def ^:no-doc fns-with-specs (utils/list-fn-specs "formform.calc"))

