(ns iomed-clj.core
  (:require
   [clojure.tools.cli :refer [parse-opts]]
   [clojure.string :as string]
   [iomed-clj.mel :as mel])
  (:gen-class))

(def TEST_API "https://test.iomed.es/tagger")
(def PROD_API "https://api.iomed.es/tagger")
(def MSG_USAGE "usage: iomed [-h] [-t] [-u URL] [-k KEY] text

Command line interface to IOMED Medical Language API (MEL).

- Visit https://console.iomed.es to obtain your api key.
- Visit https://docs.iomed.es to learn more about MEL.

positional arguments:
  text

optional arguments:
  -h, --help         show this help message and exit
  -t, --test         Use the testing platform instead of production.
  -u URL, --url URL  API url (by default https://api.iomed.es).
  -k KEY, --key KEY  API key (get it at https://console.iomed.es).")

(def ERR_TEST_URL "Error: only one of --test and --url can be specified at a time.")
(def ERR_NO_TEXT "Error: you must provide a text string.")
(def ERR_NO_KEY "Error: missing api key.
You can either set the environment variable IOMED_MEL_KEY or pass the key with --key.
Please visit https://console.iomed.es to obtain your api key.")

(def cli-options
  [["-u" "--url URL" "URL of the API."]
   ["-k" "--key KEY" "Your IOMED API key."]
   ["-t" "--test" :default false]   
   ["-h" "--help"]])

(defn exit [status msg]
  (println msg)
  (System/exit status))

(defn help
  [options]
  MSG_USAGE)

(defn -main
  "I don't do a whole lot ... yet."
  [& args]
  (let [
        {:keys [options arguments errors summary]} (parse-opts args cli-options)
        key (or (:key options) (System/getenv "IOMED_MEL_KEY"))]
    (cond
      (:help options) (exit 0 (help summary))
      (nil? key) (exit 1 ERR_NO_KEY)
      (and (:url options) (:test options)) (exit 1 ERR_TEST_URL)
      (= (count arguments) 0) (exit 1 ERR_NO_TEXT))
    (let
        [url (cond (:test options) TEST_API
                   (:url options) (:url options)
                   true PROD_API)
         text (string/join " " arguments)
         result (mel/annotate text url key true)]
      (println result))))
