(ns er-model.orgs.model
  (:require
   [clojure.string :as str]
   [plumbing.core :refer :all]
   [clj-uuid :as uuid]
   [cats.core :refer [return mlet]]
   [cats.context :refer [with-context]]
   [cats.labs.manifold :refer [deferred-context]]

   [er-cassandra.record :as r]
   [er-cassandra.model :as model]
   [er-cassandra.model.types :as t]
   [er-cassandra.model.util :as util]
   [er-cassandra.model.callbacks.search-key-callback :as skey]
   [er-cassandra.model.callbacks.sort-name-callback :as sortnm]
   [er-model.users.model :as u]))

(model/defmodel Orgs
  {:callbacks {:before-save [(t/create-updated-at-callback)]}
   :primary-table {:name :orgs
                   :key [:id]}})

(model/defmodel OrgUsers
  {:callbacks {:before-save [(t/create-updated-at-callback)
                             (skey/create-search-keys-callback
                              :search_key
                              :username
                              :name)
                             (sortnm/create-sort-name-callback
                              :sort_name
                              :name)]}
   :primary-table {:name :org_users
                   :key [[:org_id] :user_id]}
   :unique-key-tables [{:name :org_users_by_username
                        :key [:org_id :username]}]
   :lookup-key-tables [{:name :user_orgs
                        :key [[:user_id] :org_id]}
                       {:name :org_users_by_sort_name
                        :key [[:org_id] :sort_name :user_id]}
                       {:name :org_users_by_search_key
                        :key [[:org_id] :search_key :user_id]
                        :collections {:search_key :set}}]})

(defnk find-org
  [session org]
  (with-context deferred-context
    (mlet [org-id (return (t/extract-uber-key-value
                           Orgs
                           org))]
      (model/select-one session
                        Orgs
                        :id
                        org-id))))

(defnk find-for-user
  [session user]
  (with-context deferred-context
    (mlet [user-id (return (t/extract-uber-key-value
                                u/Users
                                user))
             org-users (model/select session OrgUsers :user_id user-id)
             org-ids (return (map :org_id org-users))]

      (model/select-many session
                         Orgs
                         :id
                         org-ids))))

(defnk find-org-users-for-user
  [session user]
  (let [user-id (last (t/extract-uber-key-value u/Users user))]
    (with-context deferred-context
      (mlet [user-orgs (model/select session
                                     OrgUsers
                                     [:user_id]
                                     [user-id])
             org-user-ids (return (map (fn [uo] [(:org_id uo) user-id])
                                       user-orgs))]
        (model/select-many session
                           OrgUsers
                           [:org_id :user_id]
                           org-user-ids)))))

(defnk find-org-users
  "load too many then filter down because of repetition in index table"
  [session org {q ""} {limit 20}]
  (let [org-id (last (t/extract-uber-key-value Orgs org))
        q (str/trim (or q ""))
        q2 (str q "\uffff")]
    (with-context deferred-context
      (if (empty? q)
        (model/select session
                      OrgUsers
                      [:org_id]
                      [org-id]
                      {:from :org_users_by_sort_name
                       :limit limit})
        (mlet [uf (model/select session
                                OrgUsers
                                [:org_id]
                                [org-id]
                                {:from :org_users_by_search_key
                                 :where [[:>= :search_key q]
                                         [:<= :search_key q2]]
                                 :limit (* 2 limit)})
               f (return (distinct-by :user_id uf))]
          (return (take limit f)))))))

(defnk find-many-org-users
  [session org user-ids]
  (let [org-id (t/extract-uber-key-value Orgs org)]
    (model/select-many
     session
     OrgUsers
     [:org_id :user_id]
     (map (fn [uid] [org-id uid])
          user-ids))))

(defnk add-user-to-org
  [session org user org-user]
  (let [org-id (last (t/extract-uber-key-value Orgs org))
        user-id (last (t/extract-uber-key-value u/Users user))]
    (model/upsert session OrgUsers (merge
                                    org-user
                                    {:org_id org-id
                                     :user_id user-id}))))
