(ns er-model.conversations.model
  (:require
   [plumbing.core :refer :all]
   [clj-uuid :as uuid]
   [clojure.set :as set]
   [taoensso.timbre :refer [log debug info warn]]
   [cats.core :refer [mlet return]]
   [cats.context :refer [with-context]]
   [cats.labs.manifold :refer [deferred-context]]
   [er-cassandra.record :as r]
   [er-cassandra.model :as model]
   [er-cassandra.model.types :as t]
   [er-cassandra.model.util :as util :refer [combine-responses]]
   [er-model.users.model :as u]
   [er-model.groups.model :as g]
   [er-model.orgs.model :as o]))

(model/defmodel Conversations
  {:primary-table {:name :conversations
                   :key [:org_id :id]}})

(model/defmodel ConversationAddressees
  {:primary-table {:name :conversation_addressees
                   :key [[:org_id
                          :conversation_id]
                         :addressee_type
                         :addressee_id]}
   :secondary-tables [{:name :addressee_conversations
                       :key [[:org_id
                              :addressee_id
                              :conversation_type]
                             :conversation_id]}]})

(model/defmodel ConversationParticipants
  {:primary-table {:name :conversation_participants
                   :key [[:org_id
                          :conversation_id]
                         :participant_id]}

   :secondary-tables [{:name :participant_conversations
                       :key [[:org_id
                              :participant_id
                              :conversation_type]
                             :conversation_id]}]})

(model/defmodel UserRecentConversations
  {:primary-table {:name :user_recent_conversations
                   :key [[:org_id
                          :user_id
                          :conversation_type]
                         :last_message_time
                         :conversation_id]}})

(model/defmodel ConversationMessages
  {:primary-table {:name :conversation_messages
                   :key [[:org_id
                          :conversation_id]
                         :message_time
                         :id]}})

(model/defmodel PollConversations
  {:primary-table {:name :poll_conversations
                   :key [[:org_id
                          :conversation_id]
                         :poll_type
                         :poll_time
                         :owner_id
                         :completed]}})

(model/defmodel PollConversationVotes
  {:primary-table {:name :poll_conversation_votes
                   :key [[:org_id
                          :conversation_id]
                         :user_id]}})

(defnk find-conversation
  [session
   conversation]
  (let [[org-id conversation-id] (t/extract-uber-key-value
                                  Conversations
                                  conversation)]
    (model/select-one
     session
     Conversations
     [:org_id :id]
     [org-id conversation-id])))

(defnk find-many
  [session org-id ids]
  (model/select-many
   session
   Conversations
   [:org_id :id]
   (for [id ids]
     [org-id id])))

(defnk find-recent-conversations-for-org-user-and-type
  [session
   org
   user
   conversation-type ;; can be a seq for an IN query !
   {after nil}
   {before nil}
   {limit nil}]
  (let [org-id (last (t/extract-uber-key-value o/Orgs org))
        user-id (last (t/extract-uber-key-value
                       u/Users
                       user))]
    (model/select
     session
     UserRecentConversations
     [:org_id :user_id :conversation_type]
     [org-id user-id conversation-type]
     {:order-by [:last_message_time :desc]
      :where
      (filter identity
              [(when after [:>= :last_message_time after])
               (when before [:<= :last_message_time before])])
      :limit limit})))

(defnk delete-older-recent-conversations
  [session
   org
   conversation-id
   before]
  (info ["before" before])
  (with-context deferred-context
    (mlet [rcs (er-cassandra.record/select
                session
                :user_recent_conversations
                :conversation_id conversation-id)

           sorted-rcs (return
                       (sort-by :last_message_time rcs))

           [before after] (return
                           (split-with (fn [rc]
                                         (< (compare (:last_message_time rc)
                                                     before)
                                            0))
                                       sorted-rcs))
           last-after-message-id (return (:last_message_id (last after)))

           das (return (when last-after-message-id
                         (filter (fn [urc]
                                   (not= last-after-message-id
                                         (:last_message_id urc)))
                                 after)))
           deletes (return (concat before das))]
      (info ["last-after-message-id" last-after-message-id])
      (info ["sorted" sorted-rcs])
      (er-cassandra.model.delete/delete-many
       session
       UserRecentConversations
       [:org_id :user_id :conversation_type :last_message_time :conversation_id]
       deletes)
      )))

(defnk find-messages-for-conversation
  [session
   conversation
   {after nil}
   {before nil}
   {limit nil}]
  (let [[org-id conversation-id] (t/extract-uber-key-value
                                  Conversations
                                  conversation)]
    (model/select
     session
     ConversationMessages
     [:org_id
      :conversation_id]
     [org-id conversation-id]
     {:order-by [:message_time :desc]
      :where
      (filter identity
              [(when after [:>= :message_time after])
               (when before [:<= :message_time before])])
      :limit limit})))

(defnk find-addressees-for-conversation
  "retrieve user and group addressees"
  [session
   conversation]
  (let [[org-id conversation-id] (return
                                  (t/extract-uber-key-value
                                   Conversations
                                   conversation))]
    (model/select
     session
     ConversationAddressees
     [:org_id :conversation_id]
     [org-id conversation-id])))

(defnk find-participants-for-conversation
  "find existing participant records for a conversation"
  [session
   conversation]
  (let [[org-id conversation-id] (return
                                  (t/extract-uber-key-value
                                   Conversations
                                   conversation))]
    (model/select
     session
     ConversationParticipants
     [:org_id :conversation_id]
     [org-id conversation-id])))

(defnk expand-all-participants-for-conversation
  "expand participants in a converstaion - start with addressees,
   then expand groups into users, create new (unsaved)
   ConversationParticipants for users who don't already have records
   returns [existing-cps new-cps]"
  [session
   conversation]
  (with-context deferred-context
    (mlet [[org-id conversation-id] (return
                                     (t/extract-uber-key-value Conversations
                                                               conversation))

           ;; fetch the conversation record if we only have an id...
           ;; we need the type later
           conversation-record (if (:type conversation)
                                 ;; it's already a record
                                 (return conversation)
                                 (model/select-one session Conversations
                                                   [:org_id :id]
                                                   [org-id conversation-id]))

           cads (model/select
                 session
                 ConversationAddressees
                 [:org_id :conversation_id]
                 [org-id conversation-id])

           u-ad-ids (->> cads
                         (filter (fn [p] (= "user" (:addressee_type p))))
                         (map :addressee_id)
                         set
                         return)

           grp-ad-ids (->> cads
                           (filter (fn [p] (= "group" (:addressee_type p))))
                           (map :addressee_id)
                           set
                           return)

           grp-users (->> grp-ad-ids
                          (map #(g/find-user-groups-for-group
                                 {:session session
                                 :group [org-id %]}))
                          combine-responses)

           grp-user-ids (->> grp-users
                             (map :user_id)
                             set
                             return)

           all-user-ids (return (set/union u-ad-ids grp-user-ids))

           cps (model/select
                session
                ConversationParticipants
                [:org_id :conversation_id]
                [org-id conversation-id])

           cp-ids (->> cps
                       (map :participant_id)
                       set
                       return)

           no-cp-ids (return (set/difference all-user-ids cp-ids))

           new-cps (->> no-cp-ids
                        (map (fn [u-id]
                               {:org_id org-id
                                :conversation_id conversation-id
                                :conversation_type (:type conversation-record)
                                :participant_id u-id}))
                        return)]
      (return [cps new-cps]))))

(defnk create-addressees-for-group-conversation
  [session
   group]
  (let [[org-id group-id] (t/extract-uber-key-value g/Groups group)]
    (with-context deferred-context
      (mlet [users (g/find-users-for-group {:session session
                                            :group [org-id group-id]})

             addressees (return
                         (for [u users]
                           {:org_id org-id
                            :conversation_id group-id
                            :conversation_type "group"
                            :addressee_type "user"
                            :addressee_id (:id u)}))]

        (combine-responses
         (for [a addressees]
           (model/upsert session ConversationAddressees a)))))))

(defnk create-group-conversation*
  [session
   group]
  (let [[org-id group-id] (t/extract-uber-key-value g/Groups group)]
    (with-context deferred-context
      (mlet [group (if (:name group)
                       group
                       (model/select-one
                        session
                        g/Groups
                        [:org_id :id]
                        [org-id group-id]))
               conv {:org_id org-id
                     :id group-id
                     :name (:name group)
                     :type "group"
                     :participant_ids nil}]
        (model/upsert session
                      Conversations
                      conv)
        (return conv)))))

(defnk create-group-conversation
  [session
   group :as params]
  (create-addressees-for-group-conversation params)
  (create-group-conversation* params))

(defn private-conversation-id
  [from-id to-id]
  (let [[a b] (sort [from-id to-id])]
    (uuid/v5 a b)))

(defnk create-addressees-for-private-conversation-in-org
  [session
   org
   from
   to]
  (let [org-id (last (t/extract-uber-key-value o/Orgs org))
        from-id (last (t/extract-uber-key-value u/Users from))
        to-id (last (t/extract-uber-key-value u/Users to))
        conv-id (private-conversation-id from-id to-id)
        addressees (for [u-id [from-id to-id]]
                     {:org_id org-id
                      :conversation_id conv-id
                      :conversation_type "private"
                      :addressee_type "user"
                      :addressee_id u-id})]
    (combine-responses
     (for [a addressees]
       (model/upsert session ConversationAddressees a)))))

(defnk create-private-conversation-in-org*
  [session
   org
   from
   to]
  (let [org-id (last (t/extract-uber-key-value o/Orgs org))
        from-id (last (t/extract-uber-key-value u/Users from))
        to-id (last (t/extract-uber-key-value u/Users to))
        conv-id (private-conversation-id from-id to-id)]
    (with-context deferred-context
      (mlet [from (if (:username from)
                      from
                      (model/select-one
                       session
                       u/Users
                       :id
                       from-id))

               to (if (:username to)
                    to
                    (model/select-one
                     session
                     u/Users
                     :id
                     to-id))

               conv (return {:org_id org-id
                             :id conv-id
                             :participant_ids (set [from-id to-id])
                             :name nil
                             :type "private"})

               ]
        (model/upsert session
                      Conversations
                      conv)
        (return conv)))))

(defnk create-private-conversation-in-org
  [session
   org
   from
   to :as params]
  ;; a very cassandra way - fire and forget
  (create-addressees-for-private-conversation-in-org params)
  (create-private-conversation-in-org* params))
