(ns cljsbuild.core
  (:require
    [clojure.string :as string]
    [clojure.set :as cset]
    [clj-stacktrace.repl :as st]
    [fs :as fs] 
    [cljs.closure :as cljsc]))

(defn- filter-cljs [files types]
  (let [ext #(last (string/split % #"\."))]
    (filter #(types (ext %)) files)))

(defn- find-dir-cljs [root files types]
  (for [cljs (filter-cljs files types)] (fs/join root cljs)))

(defn- find-cljs [dir types]
  (let [iter (fs/iterdir dir)]
    (mapcat
      (fn [[root _ files]]
        (find-dir-cljs root files types))
      iter)))

(defn- elapsed [started-at]
  (let [elapsed-us (- (. System (nanoTime)) started-at)]
    (with-precision 2
      (str (/ (double elapsed-us) 1000000000) " seconds"))))

(defn- compile-cljs [cljs-path compiler-options]
  (let [output-file (:output-to compiler-options)]
    (print (str "Compiling " output-file " from " cljs-path "...")) 
    (flush) 
    (fs/mkdirs (fs/dirname output-file)) 
    (let [started-at (. System (nanoTime))]
      (try
        (cljsc/build cljs-path compiler-options)
        (println (str " Done in " (elapsed started-at) "."))
        (catch Exception e
          (println " Failed!")
          (st/pst+ e))))))

(defn- is-macro-file? [file]
  (not (neg? (.indexOf (slurp file) ";*CLJSBUILD-MACRO-FILE*;"))))

; There is a little bit of madness here to share macros between Clojure
; and ClojureScript.  The latter needs a  (:require-macros ...) whereas the
; former just wants  (:require ...).  Thus, we have a ;*CLJSBUILD-REMOVE*;
; conditional comment to allow different code to be used for ClojureScript files.
(defn- filtered-crossover-file [file]
  (str
    "; DO NOT EDIT THIS FILE! IT WAS AUTOMATICALLY GENERATED BY\n"
    "; lein-cljsbuild FROM THE FOLLOWING SOURCE FILE:\n"
    "; " file "\n\n"
    (string/replace (slurp file) ";*CLJSBUILD-REMOVE*;" "")))

(defn- crossover-to [clj-path cljs-path from-file]
  (let [abspath (fs/abspath from-file)
        subpath (string/replace-first
                  (fs/abspath from-file)
                  (fs/abspath clj-path) "")
        to-file (fs/normpath
                  (fs/join (fs/abspath cljs-path) subpath))]
    (if (is-macro-file? from-file)
      to-file
      (string/replace to-file #"\.clj$" ".cljs"))))

(defmacro dofor [seq-exprs body-expr]
  `(doall (for ~seq-exprs ~body-expr)))

(defn- ns-to-path [ns]
  (let [underscored (string/replace (str ns) #"-" "_")]
    (apply fs/join
      (string/split underscored #"\."))))

(defn- fail [& args]
  (throw (Exception. (apply str args))))

(defn- copy-crossovers [clj-path cljs-path crossovers]
  (dofor [crossover crossovers]
    (do
      (when (map? crossover)
        (fail "Sorry, crossovers now need to be specified by namespace rather than the old :from-dir/:to-dir map.")) 
      (let [ns-path (ns-to-path crossover)
            clj-ns-path (fs/join clj-path ns-path)
            isdir? (fs/directory? clj-ns-path)
            clj-ns-file (str clj-ns-path ".clj")
            isfile? (fs/file? clj-ns-file)]
        (when (and (not isdir?) (not isfile?))
          (fail "Invalid crossover: " crossover))
        (let [from-files (if isdir?
                           (find-cljs clj-ns-path #{"clj"})
                           [clj-ns-file])
              to-files (map (partial crossover-to clj-path cljs-path) from-files)]
          (doseq [dir (map fs/dirname to-files)]
            (fs/mkdirs dir)) 
          (dofor [[from-file to-file] (zipmap from-files to-files)]
            (when
              (or
                (not (fs/exists? to-file))
                (> (fs/mtime from-file) (fs/mtime to-file)))
              (do
                (spit to-file (filtered-crossover-file from-file))
                :updated))))))))

(defn run-compiler [clj-path cljs-path crossovers compiler-options watch?]
  (println "Compiler started.")
  (loop [last-input-mtimes {}]
    (let [output-file (:output-to compiler-options)
          output-mtime (if (fs/exists? output-file) (fs/mtime output-file) 0)
          ; Need to return *.clj as well as *.cljs because ClojureScript
          ; macros are written in Clojure.
          input-files (find-cljs cljs-path #{"clj" "cljs"})
          input-mtimes (map fs/mtime input-files)
          crossover-updated? (some #{:updated}
                               (flatten
                                 (copy-crossovers clj-path cljs-path crossovers)))]
      (when (or
              (and
                (not= last-input-mtimes input-mtimes) 
                (some #(< output-mtime %) input-mtimes)) 
              crossover-updated?)
        (compile-cljs cljs-path compiler-options))
      (when watch?
        (Thread/sleep 100)
        (recur input-mtimes)))))
