(ns embelyon.seer.visualize
  "Provides functionality for viewing CloudFormation resource and property type
  relations via Graphviz"
  (:require [embelyon.codex.core :as codex]
            [embelyon.seer.visualize.spec :as vs]
            [embelyon.codex.spec :as cs]
            [embelyon.codex.impl.specification.spec :as ss]
            [dorothy.core :as viz]
            [dorothy.jvm :as ui]
            [clojure.spec.alpha :as s]))

;;; Data Manipulation/Access

(defn- has-type?
  [[_ item]]
  (or (contains? item :Type)
      (contains? item :ItemType)))

(defn- not-list?
  [[_ item]]
  (not= "List" (:Type item)))

(def is-complex?
  (every-pred has-type? not-list?))

(defn- has-complex-types?
  [item]
  (when (contains? item :Properties)
    (some is-complex? (:Properties item))))

(defn- with-property-meta
  "Preserves meta but adds parent id meta to produce edges
  within the graph"
  [property-type parent]
  (let [m  (meta property-type)
        pm (meta parent)]
    (with-meta property-type
      (merge m {:parent-id (:id pm)}))))

(defn- item-property-type-mapper
  "Returns a mapping function that looks a full specification for a property type
  and adds property meta to each result"
  [node opts]
  (fn [[_ item]]
    (when-let [property-type (codex/item-property-type node item opts)]
      (with-property-meta property-type node))))

(defn- complex-properties
  "Returns a function that returns property types belonging to a given node"
  [opts]
  (fn [node]
    (->> (:Properties node)
         (filter is-complex?)
         (map (item-property-type-mapper node opts)))))

(defn- flatten-dependencies
  "Navigate a resource dependency tree and return a flat representation
  of that hierarchy"
  [resource opts]
  (into [resource]
    (->> resource
         (tree-seq has-complex-types? (complex-properties opts))
         (rest)
         (remove nil?))))

;;; Presentation Functions

(defn- type-label
  "Create a string representing the AWS type"
  [item]
  (let [is-list? (= "List" (:Type item))
        type'    (some item #{:ItemType :PrimitiveType :PrimitiveItemType :Type})]
    (if is-list?
      (str ": List<" type' ">")
      (str ": " type'))))

(defn- required-label
  [item]
  (when (:Required item)
    "(required)"))

(defn- append-property-text
  "Build a string for a resource or property type's properties"
  [string [id item]]
  (str
    string
    (name id)
    (type-label item)
    " "
    (required-label item)
    "\\l"))

(defn- properties
  "Create a text representation of the properties in an aws resource or property type"
  [item]
  (let [properties (:Properties item)]
    (reduce append-property-text "" properties)))

(defn- text
  "The text content of a graphviz node"
  [id item]
  (-> id
      (name)
      (str "\n\n")
      (str (properties item))))

(defn- style
  "Sets the style of a graphviz node"
  [item attrs]
  (let [{type :type} (meta item)]
    (merge attrs
      (if (= :resource type)
        {:style "filled" :fillcolor :lightblue}
        {:style "filled" :fillcolor :lightyellow}))))

;;; Graphviz Integration

(defn- make-node
  "Creates a graphviz node"
  [id item]
  [(viz/gen-id id)
   (style item
     {:label (text id item)
      :shape "box"})])

(defn- make-relationship
  "Creates a vector associating a parent node with a child"
  [parent-id id]
  [(viz/gen-id parent-id) (viz/gen-id id)])

(defn- item->nodes
  "Convert a single resource or property type to a set of graphviz nodes"
  [item]
  (let [{parent-id :parent-id id :id} (meta item)
        node                          (make-node id item)
        rel                           (make-relationship parent-id id)]
    (if-not parent-id
      [node]
      [node rel])))

(defn- items->nodes
  "Converts a sequence of resource or property types to a flat vector of
  graphviz nodes"
  [items]
  (mapcat item->nodes items))

;;; Graph Functions

(defn service
  "Return a service as a vector of graphviz node vectors"
  ([service-name opts]
   (some->>
     (codex/resources service-name opts)
     (map #(flatten-dependencies % opts))
     (map items->nodes)))
  ([service-name]
   (service service-name {})))

(s/fdef service
  :args (s/cat :service-name ::cs/id :opts (s/? ::cs/options))
  :ret  (s/nilable ::vs/graph))

(defn resource
  "Return a resource as a graphviz node vector"
  ([id opts]
   (some->
     (codex/resource id opts)
     (flatten-dependencies opts)
     (items->nodes)))
  ([id]
   (resource id {})))

(s/fdef resource
  :args (s/cat :id ::cs/id :opts (s/? ::cs/options))
  :ret  (s/nilable ::vs/graph))

(defn property-type
  "Return a property type as a graphviz node vector"
  ([id opts]
   (some->
     (codex/property-type id opts)
     (flatten-dependencies opts)
     (items->nodes)))
  ([id]
   (property-type id {})))

(s/fdef property-type
  :args (s/cat :id ::cs/id :opts (s/? ::cs/options))
  :ret  (s/nilable ::vs/graph))

;;; Graph Display Functions

(defn show!
  [graph]
  (-> graph
      (viz/digraph)
      (viz/dot)
      (ui/show!)))

(defn render
  ([graph opts]
   (-> graph
       (viz/digraph)
       (viz/dot)
       (ui/render opts)))
  ([graph]
   (render graph {:format :svg})))

(defn save!
  ([graph file-name opts]
   (-> graph
       (viz/digraph)
       (viz/dot)
       (ui/save! file-name opts)))
  ([graph file-name]
   (save! graph file-name {:format :png})))
