(ns embelyon.seer.doc
  "Handles getting documentation for AWS CloudFormation types"
  (:require [net.cgrand.enlive-html :as html]
            [org.httpkit.client :as http]
            [io.aviso.ansi :as ansi]
            [clojure.string :as string]
            [clojure.spec.alpha :as s]
            [embelyon.codex.core :as codex]
            [embelyon.seer.doc.spec :as ds]
            [embelyon.codex.impl.specification.spec :as sp]))

(defn- slug
  [url]
  (second (re-find #"([\w-]+)[.]html" url)))

(defn- strip-whitespace
  [text]
  (-> text
      (string/replace #"\n" "")
      (string/replace #"[\s](?=[\s])" "")))

(def to-text
  (comp
    (map html/text)
    (map strip-whitespace)
    (map string/trim)
    (filter (complement empty?))))

;;; Selectors

(defn- id
  [slug]
  (keyword (str "#" slug)))

(defn- title
  [doc slug]
  (html/text (first (html/select doc [(id slug)]))))

(defn- description
  [doc slug]
  (->> (html/select doc {[(id slug)] [:h2]})
       (first)
       (drop 1)
       (drop-last)
       (transduce to-text conj)))

(defn- variablelist*
  [doc pos]
  (some->
    doc
    (html/select [:.variablelist])
    (seq)
    (nth (- pos 1) nil)))

(defn- variable
  [[term def]]
  {:term        (html/text term)
   :definition  (->> (html/select def [:p])
                     (transduce to-text conj))})

(defn- variablelist
  [doc pos]
  (some->
    (variablelist* doc pos)
    (html/select #{[:dt] [:dd]})
    (as-> items (map variable (partition 2 items)))))

(defn- getatt-id
  [slug]
  (id (str slug "-getatt")))

(defn- refdoc
  [doc slug]
  (->> (html/select doc {[(id (str slug "-ref"))] [(getatt-id slug)]})
       (first)
       (drop-last)
       (transduce to-text conj)))

(defn- getatt
  [doc slug]
  (->> (html/select doc {[(getatt-id slug)] [:.variablelist]})
       (first)
       (drop-last)
       (transduce to-text conj)))

;;; View

(defn- to-view
  "Creates a map representation of document structure"
  [doc slug url]
  {:url         url
   :title       (title doc slug)
   :description (description doc slug)
   :properties  (variablelist doc 1)
   :ref         (refdoc doc slug)
   :getatt      (getatt doc slug)
   :returns     (variablelist doc 2)})

(defn get-documented
  "Returns a documentable type from a CloudFormation specification map"
  [id data]
  (if-let [resource (get-in data [:ResourceTypes id])]
    resource
    (get-in data [:PropertyTypes id])))

(s/fdef get-documented
  :args (s/cat :id keyword? :data ::sp/cloud-formation-spec)
  :ret  (s/or :property-type ::sp/property :resource-type ::sp/resource))

(defn fetch
  "Returns a vector used for parsing documentation content of a resource or property type.
  This vector contains an enlive html resource, a unique slug for the type, and the original
  url of the type's documentation."
  [url]
  (some->
    (http/get url {:as :stream :follow-redirects true})
    (deref)
    (:body)
    (html/html-resource)
    (vector (slug url) url)))

(s/fdef fetch
  :args (s/cat :url ::url)
  :ret  ::ds/parse-input)

(defn parse
  "Parses the input vector into a view map used for printing"
  [[doc slug url]]
  (some->
    doc
    (to-view slug url)))

(s/fdef parse
  :args (s/cat :args ::ds/parse-input)
  :ret  ::ds/view)

;;; Printing

(defprotocol DocumentationPrinter
  "Prints cloud formation documentation"
  (print-url [self url])
  (print-title [self title])
  (print-block [self block])
  (print-variable [self variable])
  (print-heading [self heading])
  (print-subheading [self heading]))

(defn- println*
  [line]
  (println line)
  (println))

(defn- indent
  [string]
  (str "  " string))

(defn- print-blocks
  [printer lines]
  (doseq [line lines]
    (print-block printer line)))

(deftype DefaultPrinter []
  DocumentationPrinter
  (print-url [_ url] (println* (ansi/cyan url)))
  
  (print-title [_ title]
    (do
      (println)
      (println (ansi/bold-yellow title))))
  
  (print-block [_ block] (println* block))
  
  (print-variable [printer {:keys [term definition]}]
    (println* (ansi/magenta term))
    (->> definition
         (map indent)
         (print-blocks printer)))
  
  (print-heading [_ heading] (println* (ansi/yellow heading)))

  (print-subheading [_ heading] (println* (ansi/cyan heading))))

(def ^:dynamic *printer* (DefaultPrinter.))

(defn- print-variables
  [printer variables]
  (doseq [variable variables]
    (print-variable printer variable)))

(defn output!
  "Writes the view model out to the configured implementation of DocumentationPrinter"
  [{:keys [url title description properties returns ref getatt] :as view}]
  (print-title *printer* title)
  (print-url *printer* url)
  (print-blocks *printer* description)
  (print-heading *printer* "== Properties ==")
  (print-variables *printer* properties)
  (when returns
    (print-heading *printer* "== Return Values ==")
    (print-subheading *printer* (first ref))
    (print-blocks *printer* (next ref))
    (print-subheading *printer* (first getatt))
    (print-blocks *printer* (next getatt))
    (print-variables *printer* returns)))

(s/fdef output!
  :args (s/cat :view ::ds/view)
  :ret  nil?)

(defn doc
  ([id opts]
   (some->>
     (codex/specification-data opts)
     (get-documented (keyword id))
     (:Documentation)
     (fetch)
     (parse)
     (output!)))
  ([id]
   (doc id {})))
