(ns embelyon.seer.core
  "The seer allows you to read documentation for CloudFormation types, as well
  as visualize dependencies between CloudFormation resources and property types."
  (:require [embelyon.seer.doc :as doc]
            [embelyon.seer.visualize :as viz]
            [clojure.spec.alpha :as s]
            [embelyon.codex.spec :as cs]
            [embelyon.seer.visualize.spec :as vs]))

(defn doc
  "Lookup documentation for a resource or property type and print it. Printing
  can be configured by rebinding embelyon.seer.doc/*printer*"
  ([id opts]
   (doc/doc id opts))
  ([id]
   (doc/doc id)))

(s/fdef doc
  :args (s/cat :id   ::cs/id
               :opts (s/? ::cs/options))
  :ret  nil?)

(def ^:private service-pattern #"^[\w]+::[\w]+$")

(def ^:private property-type-pattern #"[.][\w]+$")

(def ^:private resource-pattern #"^[\w]+::[\w]+::[\w]+$")

(defn- service?
  [id]
  (not (empty? (re-find service-pattern id))))

(defn- resource?
  [id]
  (not (empty? (re-find resource-pattern id))))

(defn- property-type?
  [id]
  (not (empty? (re-find property-type-pattern id))))

(defmulti graph
  "Return a graph for the unique service, resource, or property type id given"
  (fn [id & args]
    (cond
      (service? id) :service
      (resource? id) :resource
      (property-type? id) :property-type
      :else :unknown)))

(defmethod graph :service
  ([id opts] (viz/service id opts))
  ([id] (viz/service id)))

(defmethod graph :resource
  ([id opts] (viz/resource id opts))
  ([id] (viz/resource id)))

(defmethod graph :property-type
  ([id opts] (viz/property-type id opts))
  ([id] (viz/property-type id)))

(defmethod graph :default [id & args]
  (throw
    (ex-info "Unrecognized CloudFormation id"
      {:id   id
       :opts (first args)})))

(s/fdef graph
  :args (s/cat :id ::cs/id :opts (s/? ::cs/options))
  :ret  (s/nilable ::vs/graph))

;;; Graph Display Functions

(defn show!
  "Display a CloudFormation graph"
  [graph]
  (viz/show! graph))

(s/fdef show!
  :args (s/cat :graph ::vs/graph)
  :ret  nil?)

(defn render
  "Render a CloudFormation graph to a specified format"
  ([graph opts]
   (viz/render graph opts))
  ([graph]
   (viz/render graph)))

(s/fdef render
  :args (s/cat :graph ::vs/graph :opts (s/? ::vs/options))
  :ret  ::vs/render-result)

(defn save!
  "Save a CloudFormation graph as an image"
  ([graph file-name opts]
   (viz/save! graph file-name opts))
  ([graph file-name]
   (viz/save! graph file-name)))

(s/fdef save!
  :args (s/cat :graph ::vs/graph :file-name string? :opts (s/? ::vs/options))
  :ret  string?)

;;; Shortcut Functions

(defn show-graph!
  "Create a graph of the identified CloudFormation resource, property type or service
  and then display it"
  ([id opts]
   (some->
     (graph id opts)
     (show!)))
  ([id]
   (show-graph! id {})))

(s/fdef show-graph!
  :args (s/cat :id ::cs/id :opts (s/? ::cs/options))
  :ret  nil?)

(defn render-graph
  "Create a graph of the identified CloudFormation resource, property type or service
  and then render it to a string or byte array"
  ([id codex-opts viz-opts]
   (some->
     (graph id codex-opts)
     (render viz-opts)))
  ([id codex-opts]
   (render-graph id codex-opts {:format :svg}))
  ([id]
   (render-graph id {} {:format :svg})))

(s/fdef render-graph
  :args (s/cat :id ::cs/id :codex-opts (s/? ::cs/options) :viz-opts (s/? ::vs/options))
  :ret  ::vs/render-result)

(defn save-graph!
  "Create a graph of the identified CloudFormation resource, property type or service
  and then save it to a file"
  ([id file-name codex-opts viz-opts]
   (some->
     (graph id codex-opts)
     (save! file-name viz-opts)))
  ([id file-name codex-opts]
   (save-graph! id file-name codex-opts {:format :png}))
  ([id file-name]
   (save-graph! id file-name {} {:format :png})))

(s/fdef save-graph!
  :args (s/cat :id ::cs/id :file-name string? :codex-opts (s/? ::cs/options) :viz-opts (s/? ::vs/options))
  :ret  string?)
