(ns {{ns-name}}.global
  (:require
    [clojure.core.async :as a]
    [clojure.tools.logging :as l]
    [{{ns-name}}.app :as app]
    [{{ns-name}}.config :refer [env]]
    [{{ns-name}}.env :refer [defaults]]
    [{{ns-name}}.incubator :as i]
    [{{ns-name}}.protocols :as p]
    [{{ns-name}}.routes.services.graphql :as graphql]
    [{{ns-name}}.scheduling :refer [make-scheduler-service-options]]
    [de.elbenwald.app-common.utils :as acu]
    [de.elbenwald.common.channel-consumer :as eccc]
    [de.elbenwald.common.infra :as eci]
    [de.elbenwald.common.io :as ecio]
    [de.elbenwald.common.lang :refer :all]
    [de.elbenwald.common.utils :refer :all]
    [de.elbenwald.rabbitmq.protocols :as erp]
    [de.elbenwald.security.protocols :as esp]
    [medley.core :as m]
    [mount.core :as mount]
    [plumbing.core :refer :all]
    [uncomplicate.fluokitten.core :as f]))

(mount/defstate component-factory
  :start
  ((:make-component-factory defaults) env))

(mount/defstate timer
  :start
  (eci/CreateComponent component-factory
                       :timer
                       {:error-mode :either}))

(mount/defstate ^{:on-reload :noop} rabbitmq-mgrs
  :start
  (letk [[rabbitmq] env]
    (->> rabbitmq
      (f/fmap (fnk [info]
                (eci/CreateComponent component-factory
                                     :rabbitmqem
                                     {:error-mode :either
                                      :settings info})))
      doall)))

(mount/defstate message-logger-agent
  :start
  (letk [[working-folder] env
         akeneo-repository-folder (str working-folder "/akeneo-repository")
         akeneo-repository (eci/CreateComponent component-factory
                                                :fs-repository
                                                {:error-mode :either
                                                 :folder akeneo-repository-folder})]
    (eci/make-logging-agent akeneo-repository
                            ["message-logger-agent"
                             akeneo-repository-folder])))

(mount/defstate rabbitmq-registrations
  :start
  (letk [[rabbitmq] env

         rabbitmq-exchange-infos (acu/build-rabbitmq-exchange-infos (->m rabbitmq))

         rabbitmq-registrations (acu/make-rabbitmq-registrations {:context (->m message-logger-agent
                                                                                rabbitmq-mgrs)
                                                                  :exchange-infos rabbitmq-exchange-infos})]
    rabbitmq-registrations)

  :stop
  (do
    (doseq [[registration-key rabbitmq-registration] rabbitmq-registrations]
      (l/info "rabbitmq-registrations unregister" registration-key)
      (erp/Unregister (get rabbitmq-mgrs (first registration-key))
                      (:registration rabbitmq-registration)))))

(mount/defstate app-events-ch
  :start
  (a/chan 10000
          (map identity)
          #(l/error % "app-events-ch"))

  :stop
  (a/close! app-events-ch))

(mount/defstate app
  :start
  (letk [app (eci/CreateComponent component-factory
                                  :app
                                  (->m :error-mode :either
                                       app-events-ch))]
    (eci/StartService app)
    app)

  :stop
  (eci/StopService app))

(mount/defstate channel-consumer
  :start
  (letk [channel-consumer (eccc/new-channel-consumer {})]
    (doto channel-consumer
      (eci/StartService)))

  :stop
  (eci/StopService channel-consumer))

(mount/defstate rabbitmq-subscriptions
  :start
  (letk [[rabbitmq] env

         rabbitmq-queue-infos (acu/build-rabbitmq-queue-infos (->m rabbitmq))

         rabbitmq-subscriptions (acu/make-rabbitmq-subscriptions {:context (->m message-logger-agent
                                                                                rabbitmq-mgrs)
                                                                  :queue-infos rabbitmq-queue-infos})]
    (do
      (doseq [[subscription-key rabbitmq-subscription] rabbitmq-subscriptions]
        (eci/RegisterChannel channel-consumer
                             (:channel rabbitmq-subscription)
                             (acu/make-rabbitmq-consumer-f (->m app
                                                                subscription-key))))
      rabbitmq-subscriptions))

  :stop
  (do
    (doseq [[subscription-key rabbitmq-subscription] rabbitmq-subscriptions]
      (l/info "rabbitmq-subscriptions unsubscribe" subscription-key)
      (erp/Unsubscribe (get rabbitmq-mgrs (first subscription-key))
                       (:subscription rabbitmq-subscription)))

    (doseq [[_ rabbitmq-subscription] rabbitmq-subscriptions]
      (eci/UnregisterChannel channel-consumer
                             (:channel rabbitmq-subscription)))))

(mount/defstate web-security
  :start
  (letk [[application-id
          authorization-rules] env

         access-logging (eci/CreateComponent component-factory
                                             :access-logging
                                             {:error-mode :either})
         authentication (eci/CreateComponent component-factory
                                             :authentication
                                             {:application-id application-id
                                              :error-mode :either})
         authorization (eci/CreateComponent component-factory
                                            :authorization
                                            {:error-mode :either
                                             :rules authorization-rules})]
    (eci/CreateComponent component-factory
                         :web-security
                         (->m :authorization-infos app/authorization-infos
                              :authorized-app app/authorized+app+app-base+service
                              :error-mode :either
                              access-logging
                              app
                              authentication
                              authorization))))

(mount/defstate graphql
  :start
  (eci/CreateComponent component-factory
                       :graphql
                       {:compiled-schema graphql/compiled-schema
                        :error-mode :either
                        :web-security web-security}))

(mount/defstate scheduler-service
  :start
  (letk [[working-folder] env
         scheduler-service-options (make-scheduler-service-options {:app app
                                                                    :error-mode :either
                                                                    :name "scheduler-service"
                                                                    :scheduling-infos (:scheduling-infos env)
                                                                    :timeout-msec (* 5 1000)
                                                                    :timer timer})
         scheduler-service (eci/CreateComponent component-factory :scheduler-service scheduler-service-options)]
    (eci/StartService scheduler-service)
    scheduler-service)

  :stop
  (eci/StopService scheduler-service))
