/*
 * Copyright (c) 2018, FusionAuth, All Rights Reserved
 */

"use strict";

var FusionAuthLogin = FusionAuthLogin || {};

/**
 * Creates a new FusionAuthLogin object. This pre-initializes everything and constructs the login widget.
 *
 * @param url {string} The root URL to the FusionAuth App.
 * @param applicationId {string} The application Id for this login form.
 * @constructor
 */
FusionAuthLogin = function(url, applicationId) {
  this.url = url;
  this.applicationId = applicationId;

  // this.extensions = new FusionAuthLogin.Extensions();

  this._setInitialOptions();

  this.headerImage = null;
  this.loginCallback = null;
  this.logoutCallback = null;

  // Fetch the HTML for now
  this._retrieveDialogHTML();

  // Open Login Dialog if we're not yet authenticated
  // Validate our JWT if we have one.

  // Get our Device ID ready.
  this._generateDeviceId();

  // Did the user check 'Trust this device' (remember this computer)
  this.trustedDevice = true;
};

FusionAuthLogin.constructor = FusionAuthLogin;
FusionAuthLogin.prototype = {

  /**
   * Close the dialog.
   */
  close: function() {
    this.overlay.style['display'] = '';
  },

  /**
   * Request a logout. This should invalidate your refresh token if you have one.
   */
  logout: function() {
    var xhr = new XMLHttpRequest();
    xhr.onreadystatechange = this._handleLogoutResponse.bind(this, xhr);
    xhr.open('POST', this.url + '/api/logout', true);
    xhr.send();
  },

  /**
   * Open the dialog to prompt the user to authenticate. If the user is already authenticated the login call back is invoked.
   */
  open: function() {
    if (this.authenticated) {
      console.info('try to open the dialog - if we have a callback - call it now.');
      if (this.loginCallback != null) {
        console.info('get the encoded JWT from storage and send it to the login callback');
        var encodedJWT = sessionStorage.getItem('fusionauth_access_token');
        console.info('call the callback: ' + encodedJWT);
        this.loginCallback(encodedJWT);
      }
    } else {
      this._open();
    }
  },

  /**
   * Enable or disable Gravatar support. Default is disabled.
   * @param value true or false
   * @returns {FusionAuthLogin} This.
   */
  withGravatarEnabled: function(value) {
    this.options['gravatarEnabled'] = value;
    return this;
  },

  /**
   * Add a login call back. This function will be called after the FusionAuth login has bee completed successfully.
   * @param loginCallback
   * @returns {FusionAuthLogin} This.
   */
  withLoginCallback: function(loginCallback) {
    this.loginCallback = loginCallback;
    return this;
  },

  /**
   * Add a logout call back. This function will be called after the FusionAuth logout sequence has completed.
   * @param logoutCallback
   * @returns {FusionAuthLogin} This.
   */
  withLogoutCallback: function(logoutCallback) {
    this.logoutCallback = logoutCallback;
    return this;
  },

  /**
   * This optional configuration allows you to provide a selector to identify the logout element. If provided we'll
   * capture the click event on this element and then submit the event after we've completed the FusionAuth logout sequence.
   *
   * It is not necessary to also provide a logout callback handler. If one is provided it will also be called after we fire
   * a click event on the element matching this selector.
   *
   * @param logoutSelector
   * @returns {FusionAuthLogin} This.
   */
  withLogoutSelector: function(logoutSelector) {

    try {
      var elements = document.querySelectorAll(logoutSelector);
      for (var i=0; i < elements.length; i++) {
        elements[i].addEventListener('click', this._handleLogoutClick.bind(this));
      }
    } catch (err) {
      throw new Error('Invalid selector. Please correct your selector');
    }

    return this;
  },

  /**
   * Set more than one option at a time by providing a map of key value pairs. This is considered an advanced
   * method to set options on the widget. The caller needs to know what properties are valid in the options object.
   *
   * @param {Object} options Key value pair of configuration options.
   * @returns {FusionAuthLogin} This.
   */
  withOptions: function(options) {
    if (options === null || typeof(options) === 'undefined') {
      return this;
    }

    for (var option in options) {
      if (options.hasOwnProperty(option)) {
        this.options[option] = options[option];
      }
    }
    return this;
  },

  /* ===================================================================================================================
   * Private methods
   * ===================================================================================================================*/

  _generateDeviceId: function() {
    var deviceId = localStorage.getItem('fusionauth_device_id');
    if (deviceId === null) {
      new FusionAuthLogin.Fingerprint2({
        excludeFlashFonts: true
      }).get((function(deviceId, components) {
        localStorage.setItem('fusionauth_device_id', deviceId);
      }).bind(this));
    }
  },

  _handleCloseClickEvent: function(event) {
    this.overlay.style['display'] = 'none';
    this._stopEvent(event);
  },

  _handleFormKeyPress: function(event) {
    // Enter key is the same as clicking the submit button
    if (event.keyCode === 13) {
      this._handleFormSubmit(event);
    }
  },

  _addClass: function(element, className) {
    var classes = element.className;
    if (classes.indexOf(className) === -1) {
      element.className = classes + ' ' + className;
    }
  },

  _removeClass: function(element, className) {
    var classes = element.className;
    var index = classes.indexOf(className);
    if (index !== -1) {
      element.className = classes.substr(0, index) + classes.substr(index + className.length, classes.length);
    }
  },

  _handleFormSubmit: function(event) {
    var submitButton = document.querySelector('.fusionauth-footer .fusionauth-submit');
    this._addClass(submitButton, 'hover');

    var xhr = new XMLHttpRequest();
    xhr.onreadystatechange = this._handleFormSubmitStates.bind(this, xhr);
    xhr.open('POST', this.url + '/api/login', true);
    xhr.setRequestHeader('Content-Type', 'application/json');

    var json = {
      'applicationId': this.applicationId,
      'loginId': this.loginId.value,
      'password': this.loginPassword.value
    };

    if (this.trustedDevice) {
      var browser = "unknown";
      var type = "UNKNOWN";
      if (navigator.userAgent.indexOf("Chrome") != -1) {
        browser = "Chrome";
        type = "BROWSER";
      } else if (navigator.userAgent.indexOf("Safari") != -1) {
        browser = "Safari";
        type = "BROWSER";
      } else if (navigator.userAgent.indexOf("Firefox") != -1) {
        browser = "Firefox";
        type = "BROWSER";
      } else if (navigator.userAgent.indexOf("MSIE") != -1) {
        browser = "Internet Explorer";
        type = "BROWSER";
      }

      json.device = localStorage.getItem('fusionauth_device_id');
      json.metaData = {
        'device': {
          'name': browser,
          'description': navigator.userAgent,
          lastAccessedAddress: this.ipAddress,
          type: type
        }
      };

      xhr.send(JSON.stringify(json));
    } else {
      xhr.send(JSON.stringify(json));
    }

    this._stopEvent(event);
  },

  _handleFormSubmitStates: function(xhr) {
    if (xhr.readyState === XMLHttpRequest.DONE) {
      // Login was successful
      if (xhr.status >= 200 && xhr.status <= 299) {
        sessionStorage.setItem('fusionauth_access_token', JSON.parse(xhr.responseText).token);
        this.close();

        // Call the provided callback with the token.
        if (this.loginCallback != null) {
          var encodedJWT = sessionStorage.getItem('fusionauth_access_token');
          this.loginCallback(encodedJWT);
        }

      } else {
        // Login failed. Show some messages.
        console.info(xhr.status);
      }
    }
  },

  _handleHTMLStates: function(xhr) {
    if (xhr.readyState === XMLHttpRequest.DONE) {
      if (xhr.status >= 200 && xhr.status <= 299) {
        // store my IP address
        this.ipAddress = xhr.getResponseHeader('Caller-IP');

        // A 299 indicates we're already authenticated and the response will contain the JWT
        this.authenticated = xhr.status === 299;
        if (this.authenticated) {
          console.info("already authenticated");

          this.encodedJWT = JSON.parse(xhr.responseText).token;
          return;
        }

        var html = xhr.responseText;
        document.body.insertAdjacentHTML('beforeend', html);

        this.overlay = document.querySelector('.fusionauth-overlay');
        this.form = document.querySelector('.fusionauth-box form');
        this.form.addEventListener('keyup', this._handleFormKeyPress.bind(this));
        this.submitButton = document.querySelector('.fusionauth-submit');
        this.submitButton.addEventListener('click', this._handleFormSubmit.bind(this));

        var loginForm = document.getElementById('fusionauth-login-form');
        this.loginId = loginForm.querySelector('input[name=loginId]');
        this.loginPassword = loginForm.querySelector('input[name=password]');

        this.loginId.addEventListener('blur', this._handleLoginIdBlurEvent.bind(this));

        this.headerImage = document.querySelector('.fusionauth-header img');
        this.defaultHeaderImage = window.location.protocol + '//' + window.location.host + this.headerImage.getAttribute('src');
        // this.defaultHeaderImage = this.headerImage.getAttribute('src');

        this.defaultHeaderImage = encodeURI(this.defaultHeaderImage);
        console.info(this.defaultHeaderImage);


        var signupForm = document.getElementById('fusionauth-signup-form');
        this.signupId = signupForm.querySelector('input[name=loginId]');
        this.signupPassword = signupForm.querySelector('input[name=password]');

        this.selectLogin = document.querySelector('[data-tab-id="fusionauth-login-form"]');
        this.loginTab = document.getElementById('fusionauth-login-form');
        this.selectRegistration = document.querySelector('[data-tab-id="fusionauth-signup-form"]');
        this.registrationTab = document.getElementById('fusionauth-signup-form');

        this.selectLogin.addEventListener('click', this._handleLoginFormSelect.bind(this));
        this.selectRegistration.addEventListener('click', this._handleRegistrationFormSelect.bind(this));

        this.closeButton = document.querySelector('.fusionauth-close-button');
        this.closeButton.addEventListener('click', this._handleCloseClickEvent.bind(this));
      }
    }
  },

  _handleJWTValidationStates: function(xhr) {
    if (xhr.readyState === XMLHttpRequest.DONE) {
      console.info(xhr.status);
      if (xhr.status !== 200) {
        this._open();
      } else {
        console.info('already authenticated. Call the registered callback');
      }
    }
  },

  _handleLoginFormSelect: function(event) {
    this._stopEvent(event);
    this.loginTab.style['display'] = '';
    this.registrationTab.style['display'] = 'none';
  },

  _handleLoginIdBlurEvent: function() {
    if (this.options['gravatarEnabled']) {
      var hash = FusionAuthLogin.md5(this.loginId.value);
      var size = 120;
      // Keep our original image as the default, this won't work when testing locally since localhost is not reachable.
      // var url = 'http://www.gravatar.com/avatar/' + hash + '.jpg?s=' + size + '&d=' + this.defaultHeaderImage;

      // Use the identicon as the default
      var url = 'http://www.gravatar.com/avatar/' + hash + '.jpg?s=' + size + '&d=identicon';
      this.headerImage.setAttribute('src', url);
    }
  },

  /**
   * Handle the logout click on the callers page. Allow the event to propagate.
   * @private
   */
  _handleLogoutClick: function(event) {
    this.logoutElement = event.target;
    this.logout();
    this._stopEvent(event);
  },

  _handleLogoutResponse: function(xhr) {
    if (xhr.readyState === XMLHttpRequest.DONE) {
      if (xhr.status >= 200 && xhr.status <= 299) {

        // If we have a logout element, make a [GET] AJAX request.
        // TODO : This is making a big assumption that [GET] is the proper method for the logout action.
        //        This should probably be handled in the logout callback, we can pass the href value and let the caller handle it.
        if (this.logoutElement !== null) {
          this._submitGetRequestForLogoutElement();
        }

        // If we have a logout call back, call it.
        if (this.logoutCallback !== null) {
          this.logoutCallback();
        }
      } else {
        // TODO display error messages. Failed to logout.
      }
    }
  },

  _handleRegistrationFormSelect: function(event) {
    this._stopEvent(event);
    this.loginTab.style['display'] = 'none';
    this.registrationTab.style['display'] = '';
  },

  _open: function() {
    this.overlay.style['display'] = 'flex';
  },

  _retrieveDialogHTML: function() {
    var xhr = new XMLHttpRequest();
    xhr.onreadystatechange = this._handleHTMLStates.bind(this, xhr);
    xhr.open('GET', this.url + '/api/login', true);
    xhr.send();
  },

  /**
   * Set the initial options for this widget.
   * @private
   */
  _setInitialOptions: function() {
    this.options = {
      'gravatarEnabled': false
    };
  },

  _stopEvent: function(event) {
    event.cancelBubble = true;
    if (event.stopPropagation) {
      event.stopPropagation();
    }
    if (event.preventDefault) {
      event.preventDefault();
    }
  },

  /**
   * Send a GET request to the logout URL found on the logout element.
   * @private
   */
  _submitGetRequestForLogoutElement: function() {
    var xhr = new XMLHttpRequest();
    xhr.open('GET', this.logoutElement.getAttribute('href'), true);
    xhr.send();
  },

  _validateAccessToken: function() {
    var xhr = new XMLHttpRequest();
    xhr.onreadystatechange = this._handleJWTValidationStates.bind(this, xhr);
    xhr.open('GET', this.url + '/api/jwt/validate', true);
    xhr.send();
  }
};

/* ===================================================================================================================
 * fingerprintjs2 - https://github.com/Valve/fingerprintjs2
 *
 * - Namespace FingerPrint2 --> FusionAuthLogin.FingerPrint2
 * ===================================================================================================================*/

/*
 * Fingerprintjs2 1.4.2 - Modern & flexible browser fingerprint library v2
 * https://github.com/Valve/fingerprintjs2
 * Copyright (c) 2015 Valentin Vasilyev (valentin.vasilyev@outlook.com)
 * Licensed under the MIT (http://www.opensource.org/licenses/mit-license.php) license.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL VALENTIN VASILYEV BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

FusionAuthLogin.Fingerprint2 = function(options) {
    if (!(this instanceof FusionAuthLogin.Fingerprint2)) {
      return new FusionAuthLogin.Fingerprint2(options);
    }

    var defaultOptions = {
      swfContainerId: "fingerprintjs2",
      swfPath: "flash/compiled/FontList.swf",
      detectScreenOrientation: true,
      sortPluginsFor: [/palemoon/i],
      userDefinedFonts: []
    };
    this.options = this.extend(options, defaultOptions);
    this.nativeForEach = Array.prototype.forEach;
    this.nativeMap = Array.prototype.map;
};

FusionAuthLogin.Fingerprint2.prototype = {
  extend: function(source, target) {
    if (source == null) { return target; }
    for (var k in source) {
      if(source[k] != null && target[k] !== source[k]) {
        target[k] = source[k];
      }
    }
    return target;
  },
  log: function(msg){
    if(window.console){
      console.log(msg);
    }
  },
  get: function(done){
    var keys = [];
    keys = this.userAgentKey(keys);
    keys = this.languageKey(keys);
    keys = this.colorDepthKey(keys);
    keys = this.pixelRatioKey(keys);
    keys = this.screenResolutionKey(keys);
    keys = this.availableScreenResolutionKey(keys);
    keys = this.timezoneOffsetKey(keys);
    keys = this.sessionStorageKey(keys);
    keys = this.localStorageKey(keys);
    keys = this.indexedDbKey(keys);
    keys = this.addBehaviorKey(keys);
    keys = this.openDatabaseKey(keys);
    keys = this.cpuClassKey(keys);
    keys = this.platformKey(keys);
    keys = this.doNotTrackKey(keys);
    keys = this.pluginsKey(keys);
    keys = this.canvasKey(keys);
    keys = this.webglKey(keys);
    keys = this.adBlockKey(keys);
    keys = this.hasLiedLanguagesKey(keys);
    keys = this.hasLiedResolutionKey(keys);
    keys = this.hasLiedOsKey(keys);
    keys = this.hasLiedBrowserKey(keys);
    keys = this.touchSupportKey(keys);
    var that = this;
    this.fontsKey(keys, function(newKeys){
      var values = [];
      that.each(newKeys, function(pair) {
        var value = pair.value;
        if (typeof pair.value.join !== "undefined") {
          value = pair.value.join(";");
        }
        values.push(value);
      });
      var murmur = that.x64hash128(values.join("~~~"), 31);
      return done(murmur, newKeys);
    });
  },
  userAgentKey: function(keys) {
    if(!this.options.excludeUserAgent) {
      keys.push({key: "user_agent", value: this.getUserAgent()});
    }
    return keys;
  },
  // for tests
  getUserAgent: function(){
    return navigator.userAgent;
  },
  languageKey: function(keys) {
    if(!this.options.excludeLanguage) {
      // IE 9,10 on Windows 10 does not have the `navigator.language` property any longer
      keys.push({ key: "language", value: navigator.language || navigator.userLanguage || navigator.browserLanguage || navigator.systemLanguage || "" });
    }
    return keys;
  },
  colorDepthKey: function(keys) {
    if(!this.options.excludeColorDepth) {
      keys.push({key: "color_depth", value: screen.colorDepth || -1});
    }
    return keys;
  },
  pixelRatioKey: function(keys) {
    if(!this.options.excludePixelRatio) {
      keys.push({key: "pixel_ratio", value: this.getPixelRatio()});
    }
    return keys;
  },
  getPixelRatio: function() {
    return window.devicePixelRatio || "";
  },
  screenResolutionKey: function(keys) {
    if(!this.options.excludeScreenResolution) {
      return this.getScreenResolution(keys);
    }
    return keys;
  },
  getScreenResolution: function(keys) {
    var resolution;
    if(this.options.detectScreenOrientation) {
      resolution = (screen.height > screen.width) ? [screen.height, screen.width] : [screen.width, screen.height];
    } else {
      resolution = [screen.width, screen.height];
    }
    if(typeof resolution !== "undefined") { // headless browsers
      keys.push({key: "resolution", value: resolution});
    }
    return keys;
  },
  availableScreenResolutionKey: function(keys) {
    if (!this.options.excludeAvailableScreenResolution) {
      return this.getAvailableScreenResolution(keys);
    }
    return keys;
  },
  getAvailableScreenResolution: function(keys) {
    var available;
    if(screen.availWidth && screen.availHeight) {
      if(this.options.detectScreenOrientation) {
        available = (screen.availHeight > screen.availWidth) ? [screen.availHeight, screen.availWidth] : [screen.availWidth, screen.availHeight];
      } else {
        available = [screen.availHeight, screen.availWidth];
      }
    }
    if(typeof available !== "undefined") { // headless browsers
      keys.push({key: "available_resolution", value: available});
    }
    return keys;
  },
  timezoneOffsetKey: function(keys) {
    if(!this.options.excludeTimezoneOffset) {
      keys.push({key: "timezone_offset", value: new Date().getTimezoneOffset()});
    }
    return keys;
  },
  sessionStorageKey: function(keys) {
    if(!this.options.excludeSessionStorage && this.hasSessionStorage()) {
      keys.push({key: "session_storage", value: 1});
    }
    return keys;
  },
  localStorageKey: function(keys) {
    if(!this.options.excludeSessionStorage && this.hasLocalStorage()) {
      keys.push({key: "local_storage", value: 1});
    }
    return keys;
  },
  indexedDbKey: function(keys) {
    if(!this.options.excludeIndexedDB && this.hasIndexedDB()) {
      keys.push({key: "indexed_db", value: 1});
    }
    return keys;
  },
  addBehaviorKey: function(keys) {
    //body might not be defined at this point or removed programmatically
    if(document.body && !this.options.excludeAddBehavior && document.body.addBehavior) {
      keys.push({key: "add_behavior", value: 1});
    }
    return keys;
  },
  openDatabaseKey: function(keys) {
    if(!this.options.excludeOpenDatabase && window.openDatabase) {
      keys.push({key: "open_database", value: 1});
    }
    return keys;
  },
  cpuClassKey: function(keys) {
    if(!this.options.excludeCpuClass) {
      keys.push({key: "cpu_class", value: this.getNavigatorCpuClass()});
    }
    return keys;
  },
  platformKey: function(keys) {
    if(!this.options.excludePlatform) {
      keys.push({key: "navigator_platform", value: this.getNavigatorPlatform()});
    }
    return keys;
  },
  doNotTrackKey: function(keys) {
    if(!this.options.excludeDoNotTrack) {
      keys.push({key: "do_not_track", value: this.getDoNotTrack()});
    }
    return keys;
  },
  canvasKey: function(keys) {
    if(!this.options.excludeCanvas && this.isCanvasSupported()) {
      keys.push({key: "canvas", value: this.getCanvasFp()});
    }
    return keys;
  },
  webglKey: function(keys) {
    if(this.options.excludeWebGL) {
      if(typeof NODEBUG === "undefined"){
        this.log("Skipping WebGL fingerprinting per excludeWebGL configuration option");
      }
      return keys;
    }
    if(!this.isWebGlSupported()) {
      if(typeof NODEBUG === "undefined"){
        this.log("Skipping WebGL fingerprinting because it is not supported in this browser");
      }
      return keys;
    }
    keys.push({key: "webgl", value: this.getWebglFp()});
    return keys;
  },
  adBlockKey: function(keys){
    if(!this.options.excludeAdBlock) {
      keys.push({key: "adblock", value: this.getAdBlock()});
    }
    return keys;
  },
  hasLiedLanguagesKey: function(keys){
    if(!this.options.excludeHasLiedLanguages){
      keys.push({key: "has_lied_languages", value: this.getHasLiedLanguages()});
    }
    return keys;
  },
  hasLiedResolutionKey: function(keys){
    if(!this.options.excludeHasLiedResolution){
      keys.push({key: "has_lied_resolution", value: this.getHasLiedResolution()});
    }
    return keys;
  },
  hasLiedOsKey: function(keys){
    if(!this.options.excludeHasLiedOs){
      keys.push({key: "has_lied_os", value: this.getHasLiedOs()});
    }
    return keys;
  },
  hasLiedBrowserKey: function(keys){
    if(!this.options.excludeHasLiedBrowser){
      keys.push({key: "has_lied_browser", value: this.getHasLiedBrowser()});
    }
    return keys;
  },
  fontsKey: function(keys, done) {
    if (this.options.excludeJsFonts) {
      return this.flashFontsKey(keys, done);
    }
    return this.jsFontsKey(keys, done);
  },
  // flash fonts (will increase fingerprinting time 20X to ~ 130-150ms)
  flashFontsKey: function(keys, done) {
    if(this.options.excludeFlashFonts) {
      if(typeof NODEBUG === "undefined"){
        this.log("Skipping flash fonts detection per excludeFlashFonts configuration option");
      }
      return done(keys);
    }
    // we do flash if swfobject is loaded
    if(!this.hasSwfObjectLoaded()){
      if(typeof NODEBUG === "undefined"){
        this.log("Swfobject is not detected, Flash fonts enumeration is skipped");
      }
      return done(keys);
    }
    if(!this.hasMinFlashInstalled()){
      if(typeof NODEBUG === "undefined"){
        this.log("Flash is not installed, skipping Flash fonts enumeration");
      }
      return done(keys);
    }
    if(typeof this.options.swfPath === "undefined"){
      if(typeof NODEBUG === "undefined"){
        this.log("To use Flash fonts detection, you must pass a valid swfPath option, skipping Flash fonts enumeration");
      }
      return done(keys);
    }
    this.loadSwfAndDetectFonts(function(fonts){
      keys.push({key: "swf_fonts", value: fonts.join(";")});
      done(keys);
    });
  },
  // kudos to http://www.lalit.org/lab/javascript-css-font-detect/
  jsFontsKey: function(keys, done) {
    var that = this;
    // doing js fonts detection in a pseudo-async fashion
    return setTimeout(function(){

      // a font will be compared against all the three default fonts.
      // and if it doesn't match all 3 then that font is not available.
      var baseFonts = ["monospace", "sans-serif", "serif"];

      var fontList = [
        "Andale Mono", "Arial", "Arial Black", "Arial Hebrew", "Arial MT", "Arial Narrow", "Arial Rounded MT Bold", "Arial Unicode MS",
        "Bitstream Vera Sans Mono", "Book Antiqua", "Bookman Old Style",
        "Calibri", "Cambria", "Cambria Math", "Century", "Century Gothic", "Century Schoolbook", "Comic Sans", "Comic Sans MS", "Consolas", "Courier", "Courier New",
        "Garamond", "Geneva", "Georgia",
        "Helvetica", "Helvetica Neue",
        "Impact",
        "Lucida Bright", "Lucida Calligraphy", "Lucida Console", "Lucida Fax", "LUCIDA GRANDE", "Lucida Handwriting", "Lucida Sans", "Lucida Sans Typewriter", "Lucida Sans Unicode",
        "Microsoft Sans Serif", "Monaco", "Monotype Corsiva", "MS Gothic", "MS Outlook", "MS PGothic", "MS Reference Sans Serif", "MS Sans Serif", "MS Serif", "MYRIAD", "MYRIAD PRO",
        "Palatino", "Palatino Linotype",
        "Segoe Print", "Segoe Script", "Segoe UI", "Segoe UI Light", "Segoe UI Semibold", "Segoe UI Symbol",
        "Tahoma", "Times", "Times New Roman", "Times New Roman PS", "Trebuchet MS",
        "Verdana", "Wingdings", "Wingdings 2", "Wingdings 3"
      ];
      var extendedFontList = [
        "Abadi MT Condensed Light", "Academy Engraved LET", "ADOBE CASLON PRO", "Adobe Garamond", "ADOBE GARAMOND PRO", "Agency FB", "Aharoni", "Albertus Extra Bold", "Albertus Medium", "Algerian", "Amazone BT", "American Typewriter",
        "American Typewriter Condensed", "AmerType Md BT", "Andalus", "Angsana New", "AngsanaUPC", "Antique Olive", "Aparajita", "Apple Chancery", "Apple Color Emoji", "Apple SD Gothic Neo", "Arabic Typesetting", "ARCHER",
        "ARNO PRO", "Arrus BT", "Aurora Cn BT", "AvantGarde Bk BT", "AvantGarde Md BT", "AVENIR", "Ayuthaya", "Bandy", "Bangla Sangam MN", "Bank Gothic", "BankGothic Md BT", "Baskerville",
        "Baskerville Old Face", "Batang", "BatangChe", "Bauer Bodoni", "Bauhaus 93", "Bazooka", "Bell MT", "Bembo", "Benguiat Bk BT", "Berlin Sans FB", "Berlin Sans FB Demi", "Bernard MT Condensed", "BernhardFashion BT", "BernhardMod BT", "Big Caslon", "BinnerD",
        "Blackadder ITC", "BlairMdITC TT", "Bodoni 72", "Bodoni 72 Oldstyle", "Bodoni 72 Smallcaps", "Bodoni MT", "Bodoni MT Black", "Bodoni MT Condensed", "Bodoni MT Poster Compressed",
        "Bookshelf Symbol 7", "Boulder", "Bradley Hand", "Bradley Hand ITC", "Bremen Bd BT", "Britannic Bold", "Broadway", "Browallia New", "BrowalliaUPC", "Brush Script MT", "Californian FB", "Calisto MT", "Calligrapher", "Candara",
        "CaslonOpnface BT", "Castellar", "Centaur", "Cezanne", "CG Omega", "CG Times", "Chalkboard", "Chalkboard SE", "Chalkduster", "Charlesworth", "Charter Bd BT", "Charter BT", "Chaucer",
        "ChelthmITC Bk BT", "Chiller", "Clarendon", "Clarendon Condensed", "CloisterBlack BT", "Cochin", "Colonna MT", "Constantia", "Cooper Black", "Copperplate", "Copperplate Gothic", "Copperplate Gothic Bold",
        "Copperplate Gothic Light", "CopperplGoth Bd BT", "Corbel", "Cordia New", "CordiaUPC", "Cornerstone", "Coronet", "Cuckoo", "Curlz MT", "DaunPenh", "Dauphin", "David", "DB LCD Temp", "DELICIOUS", "Denmark",
        "DFKai-SB", "Didot", "DilleniaUPC", "DIN", "DokChampa", "Dotum", "DotumChe", "Ebrima", "Edwardian Script ITC", "Elephant", "English 111 Vivace BT", "Engravers MT", "EngraversGothic BT", "Eras Bold ITC", "Eras Demi ITC", "Eras Light ITC", "Eras Medium ITC",
        "EucrosiaUPC", "Euphemia", "Euphemia UCAS", "EUROSTILE", "Exotc350 Bd BT", "FangSong", "Felix Titling", "Fixedsys", "FONTIN", "Footlight MT Light", "Forte",
        "FrankRuehl", "Fransiscan", "Freefrm721 Blk BT", "FreesiaUPC", "Freestyle Script", "French Script MT", "FrnkGothITC Bk BT", "Fruitger", "FRUTIGER",
        "Futura", "Futura Bk BT", "Futura Lt BT", "Futura Md BT", "Futura ZBlk BT", "FuturaBlack BT", "Gabriola", "Galliard BT", "Gautami", "Geeza Pro", "Geometr231 BT", "Geometr231 Hv BT", "Geometr231 Lt BT", "GeoSlab 703 Lt BT",
        "GeoSlab 703 XBd BT", "Gigi", "Gill Sans", "Gill Sans MT", "Gill Sans MT Condensed", "Gill Sans MT Ext Condensed Bold", "Gill Sans Ultra Bold", "Gill Sans Ultra Bold Condensed", "Gisha", "Gloucester MT Extra Condensed", "GOTHAM", "GOTHAM BOLD",
        "Goudy Old Style", "Goudy Stout", "GoudyHandtooled BT", "GoudyOLSt BT", "Gujarati Sangam MN", "Gulim", "GulimChe", "Gungsuh", "GungsuhChe", "Gurmukhi MN", "Haettenschweiler", "Harlow Solid Italic", "Harrington", "Heather", "Heiti SC", "Heiti TC", "HELV",
        "Herald", "High Tower Text", "Hiragino Kaku Gothic ProN", "Hiragino Mincho ProN", "Hoefler Text", "Humanst 521 Cn BT", "Humanst521 BT", "Humanst521 Lt BT", "Imprint MT Shadow", "Incised901 Bd BT", "Incised901 BT",
        "Incised901 Lt BT", "INCONSOLATA", "Informal Roman", "Informal011 BT", "INTERSTATE", "IrisUPC", "Iskoola Pota", "JasmineUPC", "Jazz LET", "Jenson", "Jester", "Jokerman", "Juice ITC", "Kabel Bk BT", "Kabel Ult BT", "Kailasa", "KaiTi", "Kalinga", "Kannada Sangam MN",
        "Kartika", "Kaufmann Bd BT", "Kaufmann BT", "Khmer UI", "KodchiangUPC", "Kokila", "Korinna BT", "Kristen ITC", "Krungthep", "Kunstler Script", "Lao UI", "Latha", "Leelawadee", "Letter Gothic", "Levenim MT", "LilyUPC", "Lithograph", "Lithograph Light", "Long Island",
        "Lydian BT", "Magneto", "Maiandra GD", "Malayalam Sangam MN", "Malgun Gothic",
        "Mangal", "Marigold", "Marion", "Marker Felt", "Market", "Marlett", "Matisse ITC", "Matura MT Script Capitals", "Meiryo", "Meiryo UI", "Microsoft Himalaya", "Microsoft JhengHei", "Microsoft New Tai Lue", "Microsoft PhagsPa", "Microsoft Tai Le",
        "Microsoft Uighur", "Microsoft YaHei", "Microsoft Yi Baiti", "MingLiU", "MingLiU_HKSCS", "MingLiU_HKSCS-ExtB", "MingLiU-ExtB", "Minion", "Minion Pro", "Miriam", "Miriam Fixed", "Mistral", "Modern", "Modern No. 20", "Mona Lisa Solid ITC TT", "Mongolian Baiti",
        "MONO", "MoolBoran", "Mrs Eaves", "MS LineDraw", "MS Mincho", "MS PMincho", "MS Reference Specialty", "MS UI Gothic", "MT Extra", "MUSEO", "MV Boli",
        "Nadeem", "Narkisim", "NEVIS", "News Gothic", "News GothicMT", "NewsGoth BT", "Niagara Engraved", "Niagara Solid", "Noteworthy", "NSimSun", "Nyala", "OCR A Extended", "Old Century", "Old English Text MT", "Onyx", "Onyx BT", "OPTIMA", "Oriya Sangam MN",
        "OSAKA", "OzHandicraft BT", "Palace Script MT", "Papyrus", "Parchment", "Party LET", "Pegasus", "Perpetua", "Perpetua Titling MT", "PetitaBold", "Pickwick", "Plantagenet Cherokee", "Playbill", "PMingLiU", "PMingLiU-ExtB",
        "Poor Richard", "Poster", "PosterBodoni BT", "PRINCETOWN LET", "Pristina", "PTBarnum BT", "Pythagoras", "Raavi", "Rage Italic", "Ravie", "Ribbon131 Bd BT", "Rockwell", "Rockwell Condensed", "Rockwell Extra Bold", "Rod", "Roman", "Sakkal Majalla",
        "Santa Fe LET", "Savoye LET", "Sceptre", "Script", "Script MT Bold", "SCRIPTINA", "Serifa", "Serifa BT", "Serifa Th BT", "ShelleyVolante BT", "Sherwood",
        "Shonar Bangla", "Showcard Gothic", "Shruti", "Signboard", "SILKSCREEN", "SimHei", "Simplified Arabic", "Simplified Arabic Fixed", "SimSun", "SimSun-ExtB", "Sinhala Sangam MN", "Sketch Rockwell", "Skia", "Small Fonts", "Snap ITC", "Snell Roundhand", "Socket",
        "Souvenir Lt BT", "Staccato222 BT", "Steamer", "Stencil", "Storybook", "Styllo", "Subway", "Swis721 BlkEx BT", "Swiss911 XCm BT", "Sylfaen", "Synchro LET", "System", "Tamil Sangam MN", "Technical", "Teletype", "Telugu Sangam MN", "Tempus Sans ITC",
        "Terminal", "Thonburi", "Traditional Arabic", "Trajan", "TRAJAN PRO", "Tristan", "Tubular", "Tunga", "Tw Cen MT", "Tw Cen MT Condensed", "Tw Cen MT Condensed Extra Bold",
        "TypoUpright BT", "Unicorn", "Univers", "Univers CE 55 Medium", "Univers Condensed", "Utsaah", "Vagabond", "Vani", "Vijaya", "Viner Hand ITC", "VisualUI", "Vivaldi", "Vladimir Script", "Vrinda", "Westminster", "WHITNEY", "Wide Latin",
        "ZapfEllipt BT", "ZapfHumnst BT", "ZapfHumnst Dm BT", "Zapfino", "Zurich BlkEx BT", "Zurich Ex BT", "ZWAdobeF"];

      if(that.options.extendedJsFonts) {
        fontList = fontList.concat(extendedFontList);
      }

      fontList = fontList.concat(that.options.userDefinedFonts);

      //we use m or w because these two characters take up the maximum width.
      // And we use a LLi so that the same matching fonts can get separated
      var testString = "mmmmmmmmmmlli";

      //we test using 72px font size, we may use any size. I guess larger the better.
      var testSize = "72px";

      var h = document.getElementsByTagName("body")[0];

      // div to load spans for the base fonts
      var baseFontsDiv = document.createElement("div");

      // div to load spans for the fonts to detect
      var fontsDiv = document.createElement("div");

      var defaultWidth = {};
      var defaultHeight = {};

      // creates a span where the fonts will be loaded
      var createSpan = function() {
        var s = document.createElement("span");
        /*
         * We need this css as in some weird browser this
         * span elements shows up for a microSec which creates a
         * bad user experience
         */
        s.style.position = "absolute";
        s.style.left = "-9999px";
        s.style.fontSize = testSize;
        s.style.lineHeight = "normal";
        s.innerHTML = testString;
        return s;
      };

      // creates a span and load the font to detect and a base font for fallback
      var createSpanWithFonts = function(fontToDetect, baseFont) {
        var s = createSpan();
        s.style.fontFamily = "'" + fontToDetect + "'," + baseFont;
        return s;
      };

      // creates spans for the base fonts and adds them to baseFontsDiv
      var initializeBaseFontsSpans = function() {
        var spans = [];
        for (var index = 0, length = baseFonts.length; index < length; index++) {
          var s = createSpan();
          s.style.fontFamily = baseFonts[index];
          baseFontsDiv.appendChild(s);
          spans.push(s);
        }
        return spans;
      };

      // creates spans for the fonts to detect and adds them to fontsDiv
      var initializeFontsSpans = function() {
        var spans = {};
        for(var i = 0, l = fontList.length; i < l; i++) {
          var fontSpans = [];
          for(var j = 0, numDefaultFonts = baseFonts.length; j < numDefaultFonts; j++) {
            var s = createSpanWithFonts(fontList[i], baseFonts[j]);
            fontsDiv.appendChild(s);
            fontSpans.push(s);
          }
          spans[fontList[i]] = fontSpans; // Stores {fontName : [spans for that font]}
        }
        return spans;
      };

      // checks if a font is available
      var isFontAvailable = function(fontSpans) {
        var detected = false;
        for(var i = 0; i < baseFonts.length; i++) {
          detected = (fontSpans[i].offsetWidth !== defaultWidth[baseFonts[i]] || fontSpans[i].offsetHeight !== defaultHeight[baseFonts[i]]);
          if(detected) {
            return detected;
          }
        }
        return detected;
      };

      // create spans for base fonts
      var baseFontsSpans = initializeBaseFontsSpans();

      // add the spans to the DOM
      h.appendChild(baseFontsDiv);

      // get the default width for the three base fonts
      for (var index = 0, length = baseFonts.length; index < length; index++) {
        defaultWidth[baseFonts[index]] = baseFontsSpans[index].offsetWidth; // width for the default font
        defaultHeight[baseFonts[index]] = baseFontsSpans[index].offsetHeight; // height for the default font
      }

      // create spans for fonts to detect
      var fontsSpans = initializeFontsSpans();

      // add all the spans to the DOM
      h.appendChild(fontsDiv);

      // check available fonts
      var available = [];
      for(var i = 0, l = fontList.length; i < l; i++) {
        if(isFontAvailable(fontsSpans[fontList[i]])) {
          available.push(fontList[i]);
        }
      }

      // remove spans from DOM
      h.removeChild(fontsDiv);
      h.removeChild(baseFontsDiv);

      keys.push({key: "js_fonts", value: available});
      done(keys);
    }, 1);
  },
  pluginsKey: function(keys) {
    if(!this.options.excludePlugins){
      if(this.isIE()){
        if(!this.options.excludeIEPlugins) {
          keys.push({key: "ie_plugins", value: this.getIEPlugins()});
        }
      } else {
        keys.push({key: "regular_plugins", value: this.getRegularPlugins()});
      }
    }
    return keys;
  },
  getRegularPlugins: function () {
    var plugins = [];
    for(var i = 0, l = navigator.plugins.length; i < l; i++) {
      plugins.push(navigator.plugins[i]);
    }
    // sorting plugins only for those user agents, that we know randomize the plugins
    // every time we try to enumerate them
    if(this.pluginsShouldBeSorted()) {
      plugins = plugins.sort(function(a, b) {
        if(a.name > b.name){ return 1; }
        if(a.name < b.name){ return -1; }
        return 0;
      });
    }
    return this.map(plugins, function (p) {
      var mimeTypes = this.map(p, function(mt){
        return [mt.type, mt.suffixes].join("~");
      }).join(",");
      return [p.name, p.description, mimeTypes].join("::");
    }, this);
  },
  getIEPlugins: function () {
    var result = [];
    if((Object.getOwnPropertyDescriptor && Object.getOwnPropertyDescriptor(window, "ActiveXObject")) || ("ActiveXObject" in window)) {
      var names = [
        "AcroPDF.PDF", // Adobe PDF reader 7+
        "Adodb.Stream",
        "AgControl.AgControl", // Silverlight
        "DevalVRXCtrl.DevalVRXCtrl.1",
        "MacromediaFlashPaper.MacromediaFlashPaper",
        "Msxml2.DOMDocument",
        "Msxml2.XMLHTTP",
        "PDF.PdfCtrl", // Adobe PDF reader 6 and earlier, brrr
        "QuickTime.QuickTime", // QuickTime
        "QuickTimeCheckObject.QuickTimeCheck.1",
        "RealPlayer",
        "RealPlayer.RealPlayer(tm) ActiveX Control (32-bit)",
        "RealVideo.RealVideo(tm) ActiveX Control (32-bit)",
        "Scripting.Dictionary",
        "SWCtl.SWCtl", // ShockWave player
        "Shell.UIHelper",
        "ShockwaveFlash.ShockwaveFlash", //flash plugin
        "Skype.Detection",
        "TDCCtl.TDCCtl",
        "WMPlayer.OCX", // Windows media player
        "rmocx.RealPlayer G2 Control",
        "rmocx.RealPlayer G2 Control.1"
      ];
      // starting to detect plugins in IE
      result = this.map(names, function(name) {
        try {
          new ActiveXObject(name); // eslint-disable-no-new
          return name;
        } catch(e) {
          return null;
        }
      });
    }
    if(navigator.plugins) {
      result = result.concat(this.getRegularPlugins());
    }
    return result;
  },
  pluginsShouldBeSorted: function () {
    var should = false;
    for(var i = 0, l = this.options.sortPluginsFor.length; i < l; i++) {
      var re = this.options.sortPluginsFor[i];
      if(navigator.userAgent.match(re)) {
        should = true;
        break;
      }
    }
    return should;
  },
  touchSupportKey: function (keys) {
    if(!this.options.excludeTouchSupport){
      keys.push({key: "touch_support", value: this.getTouchSupport()});
    }
    return keys;
  },
  hasSessionStorage: function () {
    try {
      return !!window.sessionStorage;
    } catch(e) {
      return true; // SecurityError when referencing it means it exists
    }
  },
  // https://bugzilla.mozilla.org/show_bug.cgi?id=781447
  hasLocalStorage: function () {
    try {
      return !!window.localStorage;
    } catch(e) {
      return true; // SecurityError when referencing it means it exists
    }
  },
  hasIndexedDB: function (){
    return !!window.indexedDB;
  },
  getNavigatorCpuClass: function () {
    if(navigator.cpuClass){
      return navigator.cpuClass;
    } else {
      return "unknown";
    }
  },
  getNavigatorPlatform: function () {
    if(navigator.platform) {
      return navigator.platform;
    } else {
      return "unknown";
    }
  },
  getDoNotTrack: function () {
    if(navigator.doNotTrack) {
      return navigator.doNotTrack;
    } else if (navigator.msDoNotTrack) {
      return navigator.msDoNotTrack;
    } else if (window.doNotTrack) {
      return window.doNotTrack;
    } else {
      return "unknown";
    }
  },
  // This is a crude and primitive touch screen detection.
  // It's not possible to currently reliably detect the  availability of a touch screen
  // with a JS, without actually subscribing to a touch event.
  // http://www.stucox.com/blog/you-cant-detect-a-touchscreen/
  // https://github.com/Modernizr/Modernizr/issues/548
  // method returns an array of 3 values:
  // maxTouchPoints, the success or failure of creating a TouchEvent,
  // and the availability of the 'ontouchstart' property
  getTouchSupport: function () {
    var maxTouchPoints = 0;
    var touchEvent = false;
    if(typeof navigator.maxTouchPoints !== "undefined") {
      maxTouchPoints = navigator.maxTouchPoints;
    } else if (typeof navigator.msMaxTouchPoints !== "undefined") {
      maxTouchPoints = navigator.msMaxTouchPoints;
    }
    try {
      document.createEvent("TouchEvent");
      touchEvent = true;
    } catch(_) { /* squelch */ }
    var touchStart = "ontouchstart" in window;
    return [maxTouchPoints, touchEvent, touchStart];
  },
  // https://www.browserleaks.com/canvas#how-does-it-work
  getCanvasFp: function() {
    var result = [];
    // Very simple now, need to make it more complex (geo shapes etc)
    var canvas = document.createElement("canvas");
    canvas.width = 2000;
    canvas.height = 200;
    canvas.style.display = "inline";
    var ctx = canvas.getContext("2d");
    // detect browser support of canvas winding
    // http://blogs.adobe.com/webplatform/2013/01/30/winding-rules-in-canvas/
    // https://github.com/Modernizr/Modernizr/blob/master/feature-detects/canvas/winding.js
    ctx.rect(0, 0, 10, 10);
    ctx.rect(2, 2, 6, 6);
    result.push("canvas winding:" + ((ctx.isPointInPath(5, 5, "evenodd") === false) ? "yes" : "no"));

    ctx.textBaseline = "alphabetic";
    ctx.fillStyle = "#f60";
    ctx.fillRect(125, 1, 62, 20);
    ctx.fillStyle = "#069";
    // https://github.com/Valve/fingerprintjs2/issues/66
    if(this.options.dontUseFakeFontInCanvas) {
      ctx.font = "11pt Arial";
    } else {
      ctx.font = "11pt no-real-font-123";
    }
    ctx.fillText("Cwm fjordbank glyphs vext quiz, \ud83d\ude03", 2, 15);
    ctx.fillStyle = "rgba(102, 204, 0, 0.2)";
    ctx.font = "18pt Arial";
    ctx.fillText("Cwm fjordbank glyphs vext quiz, \ud83d\ude03", 4, 45);

    // canvas blending
    // http://blogs.adobe.com/webplatform/2013/01/28/blending-features-in-canvas/
    // http://jsfiddle.net/NDYV8/16/
    ctx.globalCompositeOperation = "multiply";
    ctx.fillStyle = "rgb(255,0,255)";
    ctx.beginPath();
    ctx.arc(50, 50, 50, 0, Math.PI * 2, true);
    ctx.closePath();
    ctx.fill();
    ctx.fillStyle = "rgb(0,255,255)";
    ctx.beginPath();
    ctx.arc(100, 50, 50, 0, Math.PI * 2, true);
    ctx.closePath();
    ctx.fill();
    ctx.fillStyle = "rgb(255,255,0)";
    ctx.beginPath();
    ctx.arc(75, 100, 50, 0, Math.PI * 2, true);
    ctx.closePath();
    ctx.fill();
    ctx.fillStyle = "rgb(255,0,255)";
    // canvas winding
    // http://blogs.adobe.com/webplatform/2013/01/30/winding-rules-in-canvas/
    // http://jsfiddle.net/NDYV8/19/
    ctx.arc(75, 75, 75, 0, Math.PI * 2, true);
    ctx.arc(75, 75, 25, 0, Math.PI * 2, true);
    ctx.fill("evenodd");

    result.push("canvas fp:" + canvas.toDataURL());
    return result.join("~");
  },

  getWebglFp: function() {
    var gl;
    var fa2s = function(fa) {
      gl.clearColor(0.0, 0.0, 0.0, 1.0);
      gl.enable(gl.DEPTH_TEST);
      gl.depthFunc(gl.LEQUAL);
      gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
      return "[" + fa[0] + ", " + fa[1] + "]";
    };
    var maxAnisotropy = function(gl) {
      var anisotropy, ext = gl.getExtension("EXT_texture_filter_anisotropic") || gl.getExtension("WEBKIT_EXT_texture_filter_anisotropic") || gl.getExtension("MOZ_EXT_texture_filter_anisotropic");
      return ext ? (anisotropy = gl.getParameter(ext.MAX_TEXTURE_MAX_ANISOTROPY_EXT), 0 === anisotropy && (anisotropy = 2), anisotropy) : null;
    };
    gl = this.getWebglCanvas();
    if(!gl) { return null; }
    // WebGL fingerprinting is a combination of techniques, found in MaxMind antifraud script & Augur fingerprinting.
    // First it draws a gradient object with shaders and convers the image to the Base64 string.
    // Then it enumerates all WebGL extensions & capabilities and appends them to the Base64 string, resulting in a huge WebGL string, potentially very unique on each device
    // Since iOS supports webgl starting from version 8.1 and 8.1 runs on several graphics chips, the results may be different across ios devices, but we need to verify it.
    var result = [];
    var vShaderTemplate = "attribute vec2 attrVertex;varying vec2 varyinTexCoordinate;uniform vec2 uniformOffset;void main(){varyinTexCoordinate=attrVertex+uniformOffset;gl_Position=vec4(attrVertex,0,1);}";
    var fShaderTemplate = "precision mediump float;varying vec2 varyinTexCoordinate;void main() {gl_FragColor=vec4(varyinTexCoordinate,0,1);}";
    var vertexPosBuffer = gl.createBuffer();
    gl.bindBuffer(gl.ARRAY_BUFFER, vertexPosBuffer);
    var vertices = new Float32Array([-.2, -.9, 0, .4, -.26, 0, 0, .732134444, 0]);
    gl.bufferData(gl.ARRAY_BUFFER, vertices, gl.STATIC_DRAW);
    vertexPosBuffer.itemSize = 3;
    vertexPosBuffer.numItems = 3;
    var program = gl.createProgram(), vshader = gl.createShader(gl.VERTEX_SHADER);
    gl.shaderSource(vshader, vShaderTemplate);
    gl.compileShader(vshader);
    var fshader = gl.createShader(gl.FRAGMENT_SHADER);
    gl.shaderSource(fshader, fShaderTemplate);
    gl.compileShader(fshader);
    gl.attachShader(program, vshader);
    gl.attachShader(program, fshader);
    gl.linkProgram(program);
    gl.useProgram(program);
    program.vertexPosAttrib = gl.getAttribLocation(program, "attrVertex");
    program.offsetUniform = gl.getUniformLocation(program, "uniformOffset");
    gl.enableVertexAttribArray(program.vertexPosArray);
    gl.vertexAttribPointer(program.vertexPosAttrib, vertexPosBuffer.itemSize, gl.FLOAT, !1, 0, 0);
    gl.uniform2f(program.offsetUniform, 1, 1);
    gl.drawArrays(gl.TRIANGLE_STRIP, 0, vertexPosBuffer.numItems);
    if (gl.canvas != null) { result.push(gl.canvas.toDataURL()); }
    result.push("extensions:" + gl.getSupportedExtensions().join(";"));
    result.push("webgl aliased line width range:" + fa2s(gl.getParameter(gl.ALIASED_LINE_WIDTH_RANGE)));
    result.push("webgl aliased point size range:" + fa2s(gl.getParameter(gl.ALIASED_POINT_SIZE_RANGE)));
    result.push("webgl alpha bits:" + gl.getParameter(gl.ALPHA_BITS));
    result.push("webgl antialiasing:" + (gl.getContextAttributes().antialias ? "yes" : "no"));
    result.push("webgl blue bits:" + gl.getParameter(gl.BLUE_BITS));
    result.push("webgl depth bits:" + gl.getParameter(gl.DEPTH_BITS));
    result.push("webgl green bits:" + gl.getParameter(gl.GREEN_BITS));
    result.push("webgl max anisotropy:" + maxAnisotropy(gl));
    result.push("webgl max combined texture image units:" + gl.getParameter(gl.MAX_COMBINED_TEXTURE_IMAGE_UNITS));
    result.push("webgl max cube map texture size:" + gl.getParameter(gl.MAX_CUBE_MAP_TEXTURE_SIZE));
    result.push("webgl max fragment uniform vectors:" + gl.getParameter(gl.MAX_FRAGMENT_UNIFORM_VECTORS));
    result.push("webgl max render buffer size:" + gl.getParameter(gl.MAX_RENDERBUFFER_SIZE));
    result.push("webgl max texture image units:" + gl.getParameter(gl.MAX_TEXTURE_IMAGE_UNITS));
    result.push("webgl max texture size:" + gl.getParameter(gl.MAX_TEXTURE_SIZE));
    result.push("webgl max varying vectors:" + gl.getParameter(gl.MAX_VARYING_VECTORS));
    result.push("webgl max vertex attribs:" + gl.getParameter(gl.MAX_VERTEX_ATTRIBS));
    result.push("webgl max vertex texture image units:" + gl.getParameter(gl.MAX_VERTEX_TEXTURE_IMAGE_UNITS));
    result.push("webgl max vertex uniform vectors:" + gl.getParameter(gl.MAX_VERTEX_UNIFORM_VECTORS));
    result.push("webgl max viewport dims:" + fa2s(gl.getParameter(gl.MAX_VIEWPORT_DIMS)));
    result.push("webgl red bits:" + gl.getParameter(gl.RED_BITS));
    result.push("webgl renderer:" + gl.getParameter(gl.RENDERER));
    result.push("webgl shading language version:" + gl.getParameter(gl.SHADING_LANGUAGE_VERSION));
    result.push("webgl stencil bits:" + gl.getParameter(gl.STENCIL_BITS));
    result.push("webgl vendor:" + gl.getParameter(gl.VENDOR));
    result.push("webgl version:" + gl.getParameter(gl.VERSION));

    if (!gl.getShaderPrecisionFormat) {
      if (typeof NODEBUG === "undefined") {
        this.log("WebGL fingerprinting is incomplete, because your browser does not support getShaderPrecisionFormat");
      }
      return result.join("~");
    }

    result.push("webgl vertex shader high float precision:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_FLOAT ).precision);
    result.push("webgl vertex shader high float precision rangeMin:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_FLOAT ).rangeMin);
    result.push("webgl vertex shader high float precision rangeMax:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_FLOAT ).rangeMax);
    result.push("webgl vertex shader medium float precision:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_FLOAT ).precision);
    result.push("webgl vertex shader medium float precision rangeMin:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_FLOAT ).rangeMin);
    result.push("webgl vertex shader medium float precision rangeMax:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_FLOAT ).rangeMax);
    result.push("webgl vertex shader low float precision:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.LOW_FLOAT ).precision);
    result.push("webgl vertex shader low float precision rangeMin:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.LOW_FLOAT ).rangeMin);
    result.push("webgl vertex shader low float precision rangeMax:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.LOW_FLOAT ).rangeMax);
    result.push("webgl fragment shader high float precision:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_FLOAT ).precision);
    result.push("webgl fragment shader high float precision rangeMin:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_FLOAT ).rangeMin);
    result.push("webgl fragment shader high float precision rangeMax:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_FLOAT ).rangeMax);
    result.push("webgl fragment shader medium float precision:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_FLOAT ).precision);
    result.push("webgl fragment shader medium float precision rangeMin:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_FLOAT ).rangeMin);
    result.push("webgl fragment shader medium float precision rangeMax:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_FLOAT ).rangeMax);
    result.push("webgl fragment shader low float precision:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.LOW_FLOAT ).precision);
    result.push("webgl fragment shader low float precision rangeMin:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.LOW_FLOAT ).rangeMin);
    result.push("webgl fragment shader low float precision rangeMax:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.LOW_FLOAT ).rangeMax);
    result.push("webgl vertex shader high int precision:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_INT ).precision);
    result.push("webgl vertex shader high int precision rangeMin:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_INT ).rangeMin);
    result.push("webgl vertex shader high int precision rangeMax:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_INT ).rangeMax);
    result.push("webgl vertex shader medium int precision:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_INT ).precision);
    result.push("webgl vertex shader medium int precision rangeMin:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_INT ).rangeMin);
    result.push("webgl vertex shader medium int precision rangeMax:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_INT ).rangeMax);
    result.push("webgl vertex shader low int precision:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.LOW_INT ).precision);
    result.push("webgl vertex shader low int precision rangeMin:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.LOW_INT ).rangeMin);
    result.push("webgl vertex shader low int precision rangeMax:" + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.LOW_INT ).rangeMax);
    result.push("webgl fragment shader high int precision:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_INT ).precision);
    result.push("webgl fragment shader high int precision rangeMin:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_INT ).rangeMin);
    result.push("webgl fragment shader high int precision rangeMax:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_INT ).rangeMax);
    result.push("webgl fragment shader medium int precision:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_INT ).precision);
    result.push("webgl fragment shader medium int precision rangeMin:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_INT ).rangeMin);
    result.push("webgl fragment shader medium int precision rangeMax:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_INT ).rangeMax);
    result.push("webgl fragment shader low int precision:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.LOW_INT ).precision);
    result.push("webgl fragment shader low int precision rangeMin:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.LOW_INT ).rangeMin);
    result.push("webgl fragment shader low int precision rangeMax:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.LOW_INT ).rangeMax);
    return result.join("~");
  },
  getAdBlock: function(){
    var ads = document.createElement("div");
    ads.innerHTML = "&nbsp;";
    ads.className = "adsbox";
    var result = false;
    try {
      // body may not exist, that's why we need try/catch
      document.body.appendChild(ads);
      result = document.getElementsByClassName("adsbox")[0].offsetHeight === 0;
      document.body.removeChild(ads);
    } catch (e) {
      result = false;
    }
    return result;
  },
  getHasLiedLanguages: function(){
    //We check if navigator.language is equal to the first language of navigator.languages
    if(typeof navigator.languages !== "undefined"){
      try {
        var firstLanguages = navigator.languages[0].substr(0, 2);
        if(firstLanguages !== navigator.language.substr(0, 2)){
          return true;
        }
      } catch(err) {
        return true;
      }
    }
    return false;
  },
  getHasLiedResolution: function(){
    if(screen.width < screen.availWidth){
      return true;
    }
    if(screen.height < screen.availHeight){
      return true;
    }
    return false;
  },
  getHasLiedOs: function(){
    var userAgent = navigator.userAgent.toLowerCase();
    var oscpu = navigator.oscpu;
    var platform = navigator.platform.toLowerCase();
    var os;
    //We extract the OS from the user agent (respect the order of the if else if statement)
    if(userAgent.indexOf("windows phone") >= 0){
      os = "Windows Phone";
    } else if(userAgent.indexOf("win") >= 0){
      os = "Windows";
    } else if(userAgent.indexOf("android") >= 0){
      os = "Android";
    } else if(userAgent.indexOf("linux") >= 0){
      os = "Linux";
    } else if(userAgent.indexOf("iphone") >= 0 || userAgent.indexOf("ipad") >= 0 ){
      os = "iOS";
    } else if(userAgent.indexOf("mac") >= 0){
      os = "Mac";
    } else{
      os = "Other";
    }
    // We detect if the person uses a mobile device
    var mobileDevice;
    if (("ontouchstart" in window) ||
        (navigator.maxTouchPoints > 0) ||
        (navigator.msMaxTouchPoints > 0)) {
      mobileDevice = true;
    } else{
      mobileDevice = false;
    }

    if(mobileDevice && os !== "Windows Phone" && os !== "Android" && os !== "iOS" && os !== "Other"){
      return true;
    }

    // We compare oscpu with the OS extracted from the UA
    if(typeof oscpu !== "undefined"){
      oscpu = oscpu.toLowerCase();
      if(oscpu.indexOf("win") >= 0 && os !== "Windows" && os !== "Windows Phone"){
        return true;
      } else if(oscpu.indexOf("linux") >= 0 && os !== "Linux" && os !== "Android"){
        return true;
      } else if(oscpu.indexOf("mac") >= 0 && os !== "Mac" && os !== "iOS"){
        return true;
      } else if(oscpu.indexOf("win") === 0 && oscpu.indexOf("linux") === 0 && oscpu.indexOf("mac") >= 0 && os !== "other"){
        return true;
      }
    }

    //We compare platform with the OS extracted from the UA
    if(platform.indexOf("win") >= 0 && os !== "Windows" && os !== "Windows Phone"){
      return true;
    } else if((platform.indexOf("linux") >= 0 || platform.indexOf("android") >= 0 || platform.indexOf("pike") >= 0) && os !== "Linux" && os !== "Android"){
      return true;
    } else if((platform.indexOf("mac") >= 0 || platform.indexOf("ipad") >= 0 || platform.indexOf("ipod") >= 0 || platform.indexOf("iphone") >= 0) && os !== "Mac" && os !== "iOS"){
      return true;
    } else if(platform.indexOf("win") === 0 && platform.indexOf("linux") === 0 && platform.indexOf("mac") >= 0 && os !== "other"){
      return true;
    }

    if(typeof navigator.plugins === "undefined" && os !== "Windows" && os !== "Windows Phone"){
      //We are are in the case where the person uses ie, therefore we can infer that it's windows
      return true;
    }

    return false;
  },
  getHasLiedBrowser: function () {
    var userAgent = navigator.userAgent.toLowerCase();
    var productSub = navigator.productSub;

    //we extract the browser from the user agent (respect the order of the tests)
    var browser;
    if(userAgent.indexOf("firefox") >= 0){
      browser = "Firefox";
    } else if(userAgent.indexOf("opera") >= 0 || userAgent.indexOf("opr") >= 0){
      browser = "Opera";
    } else if(userAgent.indexOf("chrome") >= 0){
      browser = "Chrome";
    } else if(userAgent.indexOf("safari") >= 0){
      browser = "Safari";
    } else if(userAgent.indexOf("trident") >= 0){
      browser = "Internet Explorer";
    } else{
      browser = "Other";
    }

    if((browser === "Chrome" || browser === "Safari" || browser === "Opera") && productSub !== "20030107"){
      return true;
    }

    var tempRes = eval.toString().length;
    if(tempRes === 37 && browser !== "Safari" && browser !== "Firefox" && browser !== "Other"){
      return true;
    } else if(tempRes === 39 && browser !== "Internet Explorer" && browser !== "Other"){
      return true;
    } else if(tempRes === 33 && browser !== "Chrome" && browser !== "Opera" && browser !== "Other"){
      return true;
    }

    //We create an error to see how it is handled
    var errFirefox;
    try {
      throw "a";
    } catch(err){
      try{
        err.toSource();
        errFirefox = true;
      } catch(errOfErr){
        errFirefox = false;
      }
    }
    if(errFirefox && browser !== "Firefox" && browser !== "Other"){
      return true;
    }
    return false;
  },
  isCanvasSupported: function () {
    var elem = document.createElement("canvas");
    return !!(elem.getContext && elem.getContext("2d"));
  },
  isWebGlSupported: function() {
    // code taken from Modernizr
    if (!this.isCanvasSupported()) {
      return false;
    }

    var canvas = document.createElement("canvas"),
        glContext;

    try {
      glContext = canvas.getContext && (canvas.getContext("webgl") || canvas.getContext("experimental-webgl"));
    } catch(e) {
      glContext = false;
    }

    return !!window.WebGLRenderingContext && !!glContext;
  },
  isIE: function () {
    if(navigator.appName === "Microsoft Internet Explorer") {
      return true;
    } else if(navigator.appName === "Netscape" && /Trident/.test(navigator.userAgent)) { // IE 11
      return true;
    }
    return false;
  },
  hasSwfObjectLoaded: function(){
    return typeof window.swfobject !== "undefined";
  },
  hasMinFlashInstalled: function () {
    return swfobject.hasFlashPlayerVersion("9.0.0");
  },
  addFlashDivNode: function() {
    var node = document.createElement("div");
    node.setAttribute("id", this.options.swfContainerId);
    document.body.appendChild(node);
  },
  loadSwfAndDetectFonts: function(done) {
    var hiddenCallback = "___fp_swf_loaded";
    window[hiddenCallback] = function(fonts) {
      done(fonts);
    };
    var id = this.options.swfContainerId;
    this.addFlashDivNode();
    var flashvars = { onReady: hiddenCallback};
    var flashparams = { allowScriptAccess: "always", menu: "false" };
    swfobject.embedSWF(this.options.swfPath, id, "1", "1", "9.0.0", false, flashvars, flashparams, {});
  },
  getWebglCanvas: function() {
    var canvas = document.createElement("canvas");
    var gl = null;
    try {
      gl = canvas.getContext("webgl") || canvas.getContext("experimental-webgl");
    } catch(e) { /* squelch */ }
    if (!gl) { gl = null; }
    return gl;
  },
  each: function (obj, iterator, context) {
    if (obj === null) {
      return;
    }
    if (this.nativeForEach && obj.forEach === this.nativeForEach) {
      obj.forEach(iterator, context);
    } else if (obj.length === +obj.length) {
      for (var i = 0, l = obj.length; i < l; i++) {
        if (iterator.call(context, obj[i], i, obj) === {}) { return; }
      }
    } else {
      for (var key in obj) {
        if (obj.hasOwnProperty(key)) {
          if (iterator.call(context, obj[key], key, obj) === {}) { return; }
        }
      }
    }
  },

  map: function(obj, iterator, context) {
    var results = [];
    // Not using strict equality so that this acts as a
    // shortcut to checking for `null` and `undefined`.
    if (obj == null) { return results; }
    if (this.nativeMap && obj.map === this.nativeMap) { return obj.map(iterator, context); }
    this.each(obj, function(value, index, list) {
      results[results.length] = iterator.call(context, value, index, list);
    });
    return results;
  },

  /// MurmurHash3 related functions

  //
  // Given two 64bit ints (as an array of two 32bit ints) returns the two
  // added together as a 64bit int (as an array of two 32bit ints).
  //
  x64Add: function(m, n) {
    m = [m[0] >>> 16, m[0] & 0xffff, m[1] >>> 16, m[1] & 0xffff];
    n = [n[0] >>> 16, n[0] & 0xffff, n[1] >>> 16, n[1] & 0xffff];
    var o = [0, 0, 0, 0];
    o[3] += m[3] + n[3];
    o[2] += o[3] >>> 16;
    o[3] &= 0xffff;
    o[2] += m[2] + n[2];
    o[1] += o[2] >>> 16;
    o[2] &= 0xffff;
    o[1] += m[1] + n[1];
    o[0] += o[1] >>> 16;
    o[1] &= 0xffff;
    o[0] += m[0] + n[0];
    o[0] &= 0xffff;
    return [(o[0] << 16) | o[1], (o[2] << 16) | o[3]];
  },

  //
  // Given two 64bit ints (as an array of two 32bit ints) returns the two
  // multiplied together as a 64bit int (as an array of two 32bit ints).
  //
  x64Multiply: function(m, n) {
    m = [m[0] >>> 16, m[0] & 0xffff, m[1] >>> 16, m[1] & 0xffff];
    n = [n[0] >>> 16, n[0] & 0xffff, n[1] >>> 16, n[1] & 0xffff];
    var o = [0, 0, 0, 0];
    o[3] += m[3] * n[3];
    o[2] += o[3] >>> 16;
    o[3] &= 0xffff;
    o[2] += m[2] * n[3];
    o[1] += o[2] >>> 16;
    o[2] &= 0xffff;
    o[2] += m[3] * n[2];
    o[1] += o[2] >>> 16;
    o[2] &= 0xffff;
    o[1] += m[1] * n[3];
    o[0] += o[1] >>> 16;
    o[1] &= 0xffff;
    o[1] += m[2] * n[2];
    o[0] += o[1] >>> 16;
    o[1] &= 0xffff;
    o[1] += m[3] * n[1];
    o[0] += o[1] >>> 16;
    o[1] &= 0xffff;
    o[0] += (m[0] * n[3]) + (m[1] * n[2]) + (m[2] * n[1]) + (m[3] * n[0]);
    o[0] &= 0xffff;
    return [(o[0] << 16) | o[1], (o[2] << 16) | o[3]];
  },
  //
  // Given a 64bit int (as an array of two 32bit ints) and an int
  // representing a number of bit positions, returns the 64bit int (as an
  // array of two 32bit ints) rotated left by that number of positions.
  //
  x64Rotl: function(m, n) {
    n %= 64;
    if (n === 32) {
      return [m[1], m[0]];
    }
    else if (n < 32) {
      return [(m[0] << n) | (m[1] >>> (32 - n)), (m[1] << n) | (m[0] >>> (32 - n))];
    }
    else {
      n -= 32;
      return [(m[1] << n) | (m[0] >>> (32 - n)), (m[0] << n) | (m[1] >>> (32 - n))];
    }
  },
  //
  // Given a 64bit int (as an array of two 32bit ints) and an int
  // representing a number of bit positions, returns the 64bit int (as an
  // array of two 32bit ints) shifted left by that number of positions.
  //
  x64LeftShift: function(m, n) {
    n %= 64;
    if (n === 0) {
      return m;
    }
    else if (n < 32) {
      return [(m[0] << n) | (m[1] >>> (32 - n)), m[1] << n];
    }
    else {
      return [m[1] << (n - 32), 0];
    }
  },
  //
  // Given two 64bit ints (as an array of two 32bit ints) returns the two
  // xored together as a 64bit int (as an array of two 32bit ints).
  //
  x64Xor: function(m, n) {
    return [m[0] ^ n[0], m[1] ^ n[1]];
  },
  //
  // Given a block, returns murmurHash3's final x64 mix of that block.
  // (`[0, h[0] >>> 1]` is a 33 bit unsigned right shift. This is the
  // only place where we need to right shift 64bit ints.)
  //
  x64Fmix: function(h) {
    h = this.x64Xor(h, [0, h[0] >>> 1]);
    h = this.x64Multiply(h, [0xff51afd7, 0xed558ccd]);
    h = this.x64Xor(h, [0, h[0] >>> 1]);
    h = this.x64Multiply(h, [0xc4ceb9fe, 0x1a85ec53]);
    h = this.x64Xor(h, [0, h[0] >>> 1]);
    return h;
  },

  //
  // Given a string and an optional seed as an int, returns a 128 bit
  // hash using the x64 flavor of MurmurHash3, as an unsigned hex.
  //
  x64hash128: function (key, seed) {
    key = key || "";
    seed = seed || 0;
    var remainder = key.length % 16;
    var bytes = key.length - remainder;
    var h1 = [0, seed];
    var h2 = [0, seed];
    var k1 = [0, 0];
    var k2 = [0, 0];
    var c1 = [0x87c37b91, 0x114253d5];
    var c2 = [0x4cf5ad43, 0x2745937f];
    for (var i = 0; i < bytes; i = i + 16) {
      k1 = [((key.charCodeAt(i + 4) & 0xff)) | ((key.charCodeAt(i + 5) & 0xff) << 8) | ((key.charCodeAt(i + 6) & 0xff) << 16) | ((key.charCodeAt(i + 7) & 0xff) << 24), ((key.charCodeAt(i) & 0xff)) | ((key.charCodeAt(i + 1) & 0xff) << 8) | ((key.charCodeAt(i + 2) & 0xff) << 16) | ((key.charCodeAt(i + 3) & 0xff) << 24)];
      k2 = [((key.charCodeAt(i + 12) & 0xff)) | ((key.charCodeAt(i + 13) & 0xff) << 8) | ((key.charCodeAt(i + 14) & 0xff) << 16) | ((key.charCodeAt(i + 15) & 0xff) << 24), ((key.charCodeAt(i + 8) & 0xff)) | ((key.charCodeAt(i + 9) & 0xff) << 8) | ((key.charCodeAt(i + 10) & 0xff) << 16) | ((key.charCodeAt(i + 11) & 0xff) << 24)];
      k1 = this.x64Multiply(k1, c1);
      k1 = this.x64Rotl(k1, 31);
      k1 = this.x64Multiply(k1, c2);
      h1 = this.x64Xor(h1, k1);
      h1 = this.x64Rotl(h1, 27);
      h1 = this.x64Add(h1, h2);
      h1 = this.x64Add(this.x64Multiply(h1, [0, 5]), [0, 0x52dce729]);
      k2 = this.x64Multiply(k2, c2);
      k2 = this.x64Rotl(k2, 33);
      k2 = this.x64Multiply(k2, c1);
      h2 = this.x64Xor(h2, k2);
      h2 = this.x64Rotl(h2, 31);
      h2 = this.x64Add(h2, h1);
      h2 = this.x64Add(this.x64Multiply(h2, [0, 5]), [0, 0x38495ab5]);
    }
    k1 = [0, 0];
    k2 = [0, 0];
    switch(remainder) {
      case 15:
        k2 = this.x64Xor(k2, this.x64LeftShift([0, key.charCodeAt(i + 14)], 48));
      case 14:
        k2 = this.x64Xor(k2, this.x64LeftShift([0, key.charCodeAt(i + 13)], 40));
      case 13:
        k2 = this.x64Xor(k2, this.x64LeftShift([0, key.charCodeAt(i + 12)], 32));
      case 12:
        k2 = this.x64Xor(k2, this.x64LeftShift([0, key.charCodeAt(i + 11)], 24));
      case 11:
        k2 = this.x64Xor(k2, this.x64LeftShift([0, key.charCodeAt(i + 10)], 16));
      case 10:
        k2 = this.x64Xor(k2, this.x64LeftShift([0, key.charCodeAt(i + 9)], 8));
      case 9:
        k2 = this.x64Xor(k2, [0, key.charCodeAt(i + 8)]);
        k2 = this.x64Multiply(k2, c2);
        k2 = this.x64Rotl(k2, 33);
        k2 = this.x64Multiply(k2, c1);
        h2 = this.x64Xor(h2, k2);
      case 8:
        k1 = this.x64Xor(k1, this.x64LeftShift([0, key.charCodeAt(i + 7)], 56));
      case 7:
        k1 = this.x64Xor(k1, this.x64LeftShift([0, key.charCodeAt(i + 6)], 48));
      case 6:
        k1 = this.x64Xor(k1, this.x64LeftShift([0, key.charCodeAt(i + 5)], 40));
      case 5:
        k1 = this.x64Xor(k1, this.x64LeftShift([0, key.charCodeAt(i + 4)], 32));
      case 4:
        k1 = this.x64Xor(k1, this.x64LeftShift([0, key.charCodeAt(i + 3)], 24));
      case 3:
        k1 = this.x64Xor(k1, this.x64LeftShift([0, key.charCodeAt(i + 2)], 16));
      case 2:
        k1 = this.x64Xor(k1, this.x64LeftShift([0, key.charCodeAt(i + 1)], 8));
      case 1:
        k1 = this.x64Xor(k1, [0, key.charCodeAt(i)]);
        k1 = this.x64Multiply(k1, c1);
        k1 = this.x64Rotl(k1, 31);
        k1 = this.x64Multiply(k1, c2);
        h1 = this.x64Xor(h1, k1);
    }
    h1 = this.x64Xor(h1, [0, key.length]);
    h2 = this.x64Xor(h2, [0, key.length]);
    h1 = this.x64Add(h1, h2);
    h2 = this.x64Add(h2, h1);
    h1 = this.x64Fmix(h1);
    h2 = this.x64Fmix(h2);
    h1 = this.x64Add(h1, h2);
    h2 = this.x64Add(h2, h1);
    return ("00000000" + (h1[0] >>> 0).toString(16)).slice(-8) + ("00000000" + (h1[1] >>> 0).toString(16)).slice(-8) + ("00000000" + (h2[0] >>> 0).toString(16)).slice(-8) + ("00000000" + (h2[1] >>> 0).toString(16)).slice(-8);
  }
};
FusionAuthLogin.Fingerprint2.VERSION = "1.4.2";

/* ===================================================================================================================
 * JavaScript MD5 - https://github.com/blueimp/JavaScript-MD5
 *
 * Namespace MD5 to FusionAuthLogin.md5
 * ===================================================================================================================*/

/*
 * JavaScript MD5
 * https://github.com/blueimp/JavaScript-MD5
 *
 * Copyright 2011, Sebastian Tschan
 * https://blueimp.net
 *
 * Licensed under the MIT license:
 * http://www.opensource.org/licenses/MIT
 *
 * Based on
 * A JavaScript implementation of the RSA Data Security, Inc. MD5 Message
 * Digest Algorithm, as defined in RFC 1321.
 * Version 2.2 Copyright (C) Paul Johnston 1999 - 2009
 * Other contributors: Greg Holt, Andrew Kepert, Ydnar, Lostinet
 * Distributed under the BSD License
 * See http://pajhome.org.uk/crypt/md5 for more info.
 */
FusionAuthLogin.md5 = function(string) {

    /*
     * Add integers, wrapping at 2^32. This uses 16-bit operations internally
     * to work around bugs in some JS interpreters.
     */
    function safeAdd (x, y) {
      var lsw = (x & 0xFFFF) + (y & 0xFFFF)
      var msw = (x >> 16) + (y >> 16) + (lsw >> 16)
      return (msw << 16) | (lsw & 0xFFFF)
    }

    /*
     * Bitwise rotate a 32-bit number to the left.
     */
    function bitRotateLeft (num, cnt) {
      return (num << cnt) | (num >>> (32 - cnt))
    }

    /*
     * These functions implement the four basic operations the algorithm uses.
     */
    function md5cmn (q, a, b, x, s, t) {
      return safeAdd(bitRotateLeft(safeAdd(safeAdd(a, q), safeAdd(x, t)), s), b)
    }
    function md5ff (a, b, c, d, x, s, t) {
      return md5cmn((b & c) | ((~b) & d), a, b, x, s, t)
    }
    function md5gg (a, b, c, d, x, s, t) {
      return md5cmn((b & d) | (c & (~d)), a, b, x, s, t)
    }
    function md5hh (a, b, c, d, x, s, t) {
      return md5cmn(b ^ c ^ d, a, b, x, s, t)
    }
    function md5ii (a, b, c, d, x, s, t) {
      return md5cmn(c ^ (b | (~d)), a, b, x, s, t)
    }

    /*
     * Calculate the MD5 of an array of little-endian words, and a bit length.
     */
    function binlMD5 (x, len) {
      /* append padding */
      x[len >> 5] |= 0x80 << (len % 32)
      x[(((len + 64) >>> 9) << 4) + 14] = len

      var i
      var olda
      var oldb
      var oldc
      var oldd
      var a = 1732584193
      var b = -271733879
      var c = -1732584194
      var d = 271733878

      for (i = 0; i < x.length; i += 16) {
        olda = a
        oldb = b
        oldc = c
        oldd = d

        a = md5ff(a, b, c, d, x[i], 7, -680876936)
        d = md5ff(d, a, b, c, x[i + 1], 12, -389564586)
        c = md5ff(c, d, a, b, x[i + 2], 17, 606105819)
        b = md5ff(b, c, d, a, x[i + 3], 22, -1044525330)
        a = md5ff(a, b, c, d, x[i + 4], 7, -176418897)
        d = md5ff(d, a, b, c, x[i + 5], 12, 1200080426)
        c = md5ff(c, d, a, b, x[i + 6], 17, -1473231341)
        b = md5ff(b, c, d, a, x[i + 7], 22, -45705983)
        a = md5ff(a, b, c, d, x[i + 8], 7, 1770035416)
        d = md5ff(d, a, b, c, x[i + 9], 12, -1958414417)
        c = md5ff(c, d, a, b, x[i + 10], 17, -42063)
        b = md5ff(b, c, d, a, x[i + 11], 22, -1990404162)
        a = md5ff(a, b, c, d, x[i + 12], 7, 1804603682)
        d = md5ff(d, a, b, c, x[i + 13], 12, -40341101)
        c = md5ff(c, d, a, b, x[i + 14], 17, -1502002290)
        b = md5ff(b, c, d, a, x[i + 15], 22, 1236535329)

        a = md5gg(a, b, c, d, x[i + 1], 5, -165796510)
        d = md5gg(d, a, b, c, x[i + 6], 9, -1069501632)
        c = md5gg(c, d, a, b, x[i + 11], 14, 643717713)
        b = md5gg(b, c, d, a, x[i], 20, -373897302)
        a = md5gg(a, b, c, d, x[i + 5], 5, -701558691)
        d = md5gg(d, a, b, c, x[i + 10], 9, 38016083)
        c = md5gg(c, d, a, b, x[i + 15], 14, -660478335)
        b = md5gg(b, c, d, a, x[i + 4], 20, -405537848)
        a = md5gg(a, b, c, d, x[i + 9], 5, 568446438)
        d = md5gg(d, a, b, c, x[i + 14], 9, -1019803690)
        c = md5gg(c, d, a, b, x[i + 3], 14, -187363961)
        b = md5gg(b, c, d, a, x[i + 8], 20, 1163531501)
        a = md5gg(a, b, c, d, x[i + 13], 5, -1444681467)
        d = md5gg(d, a, b, c, x[i + 2], 9, -51403784)
        c = md5gg(c, d, a, b, x[i + 7], 14, 1735328473)
        b = md5gg(b, c, d, a, x[i + 12], 20, -1926607734)

        a = md5hh(a, b, c, d, x[i + 5], 4, -378558)
        d = md5hh(d, a, b, c, x[i + 8], 11, -2022574463)
        c = md5hh(c, d, a, b, x[i + 11], 16, 1839030562)
        b = md5hh(b, c, d, a, x[i + 14], 23, -35309556)
        a = md5hh(a, b, c, d, x[i + 1], 4, -1530992060)
        d = md5hh(d, a, b, c, x[i + 4], 11, 1272893353)
        c = md5hh(c, d, a, b, x[i + 7], 16, -155497632)
        b = md5hh(b, c, d, a, x[i + 10], 23, -1094730640)
        a = md5hh(a, b, c, d, x[i + 13], 4, 681279174)
        d = md5hh(d, a, b, c, x[i], 11, -358537222)
        c = md5hh(c, d, a, b, x[i + 3], 16, -722521979)
        b = md5hh(b, c, d, a, x[i + 6], 23, 76029189)
        a = md5hh(a, b, c, d, x[i + 9], 4, -640364487)
        d = md5hh(d, a, b, c, x[i + 12], 11, -421815835)
        c = md5hh(c, d, a, b, x[i + 15], 16, 530742520)
        b = md5hh(b, c, d, a, x[i + 2], 23, -995338651)

        a = md5ii(a, b, c, d, x[i], 6, -198630844)
        d = md5ii(d, a, b, c, x[i + 7], 10, 1126891415)
        c = md5ii(c, d, a, b, x[i + 14], 15, -1416354905)
        b = md5ii(b, c, d, a, x[i + 5], 21, -57434055)
        a = md5ii(a, b, c, d, x[i + 12], 6, 1700485571)
        d = md5ii(d, a, b, c, x[i + 3], 10, -1894986606)
        c = md5ii(c, d, a, b, x[i + 10], 15, -1051523)
        b = md5ii(b, c, d, a, x[i + 1], 21, -2054922799)
        a = md5ii(a, b, c, d, x[i + 8], 6, 1873313359)
        d = md5ii(d, a, b, c, x[i + 15], 10, -30611744)
        c = md5ii(c, d, a, b, x[i + 6], 15, -1560198380)
        b = md5ii(b, c, d, a, x[i + 13], 21, 1309151649)
        a = md5ii(a, b, c, d, x[i + 4], 6, -145523070)
        d = md5ii(d, a, b, c, x[i + 11], 10, -1120210379)
        c = md5ii(c, d, a, b, x[i + 2], 15, 718787259)
        b = md5ii(b, c, d, a, x[i + 9], 21, -343485551)

        a = safeAdd(a, olda)
        b = safeAdd(b, oldb)
        c = safeAdd(c, oldc)
        d = safeAdd(d, oldd)
      }
      return [a, b, c, d]
    }

    /*
     * Convert an array of little-endian words to a string
     */
    function binl2rstr (input) {
      var i
      var output = ''
      var length32 = input.length * 32
      for (i = 0; i < length32; i += 8) {
        output += String.fromCharCode((input[i >> 5] >>> (i % 32)) & 0xFF)
      }
      return output
    }

    /*
     * Convert a raw string to an array of little-endian words
     * Characters >255 have their high-byte silently ignored.
     */
    function rstr2binl (input) {
      var i
      var output = []
      output[(input.length >> 2) - 1] = undefined
      for (i = 0; i < output.length; i += 1) {
        output[i] = 0
      }
      var length8 = input.length * 8
      for (i = 0; i < length8; i += 8) {
        output[i >> 5] |= (input.charCodeAt(i / 8) & 0xFF) << (i % 32)
      }
      return output
    }

    /*
     * Calculate the MD5 of a raw string
     */
    function rstrMD5 (s) {
      return binl2rstr(binlMD5(rstr2binl(s), s.length * 8))
    }

    /*
     * Calculate the HMAC-MD5, of a key and some data (raw strings)
     */
    function rstrHMACMD5 (key, data) {
      var i
      var bkey = rstr2binl(key)
      var ipad = []
      var opad = []
      var hash
      ipad[15] = opad[15] = undefined
      if (bkey.length > 16) {
        bkey = binlMD5(bkey, key.length * 8)
      }
      for (i = 0; i < 16; i += 1) {
        ipad[i] = bkey[i] ^ 0x36363636
        opad[i] = bkey[i] ^ 0x5C5C5C5C
      }
      hash = binlMD5(ipad.concat(rstr2binl(data)), 512 + data.length * 8)
      return binl2rstr(binlMD5(opad.concat(hash), 512 + 128))
    }

    /*
     * Convert a raw string to a hex string
     */
    function rstr2hex (input) {
      var hexTab = '0123456789abcdef'
      var output = ''
      var x
      var i
      for (i = 0; i < input.length; i += 1) {
        x = input.charCodeAt(i)
        output += hexTab.charAt((x >>> 4) & 0x0F) +
            hexTab.charAt(x & 0x0F)
      }
      return output
    }

    /*
     * Encode a string as utf-8
     */
    function str2rstrUTF8 (input) {
      return unescape(encodeURIComponent(input))
    }

    /*
     * Take string arguments and return either raw or hex encoded strings
     */
    function rawMD5 (s) {
      return rstrMD5(str2rstrUTF8(s))
    }
    function hexMD5 (s) {
      return rstr2hex(rawMD5(s))
    }
    function rawHMACMD5 (k, d) {
      return rstrHMACMD5(str2rstrUTF8(k), str2rstrUTF8(d))
    }
    function hexHMACMD5 (k, d) {
      return rstr2hex(rawHMACMD5(k, d))
    }

    return rstr2hex(rawMD5(string));
};

/* ===================================================================================================================
 * Polyfill
 * ===================================================================================================================*/

// Required for fingerprintjs2. Array.prototype.indexOf()
// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/indexOf
if (!Array.prototype.indexOf) {
  Array.prototype.indexOf = function(searchElement, fromIndex) {
    var k;
    if (this == null) {
      throw new TypeError("'this' is null or undefined");
    }
    var O = Object(this);
    var len = O.length >>> 0;
    if (len === 0) {
      return -1;
    }
    var n = +fromIndex || 0;
    if (Math.abs(n) === Infinity) {
      n = 0;
    }
    if (n >= len) {
      return -1;
    }
    k = Math.max(n >= 0 ? n : len - Math.abs(n), 0);
    while (k < len) {
      if (k in O && O[k] === searchElement) {
        return k;
      }
      k++;
    }
    return -1;
  };
}