(ns elasticsearch.document
  (:refer-clojure :exclude [get delete count update])
  (:require
   [elasticsearch.common :refer [format-uri request]]
   [elasticsearch.connection.http :as hconn]
   [elasticsearch.connection :as conn]
   [slingshot.slingshot :refer [try+ throw+]]))

(defn index
  ([conn index req]
   (request conn :post (format-uri index "_doc") req))
  ([conn index id req]
   (request conn :put (format-uri index "_doc" id) req)))

(defn exists
  ([conn idx id]
   (exists conn idx id {}))
  ([conn idx id req]
   (try+
    (request conn :head (format-uri idx id) req)
    true
    (catch [:status 404] _
      false))))

(defn bulk
  ([conn req]
   (bulk conn nil req))
  ([conn idx req]
   (let [res (request conn :post (format-uri idx "_bulk") req)]
     (when (:errors res)
       ;; A response that contains an error looks like:
       ;;
       ;; {:took 17,
       ;;  :errors true,
       ;;  :items
       ;;  [{:delete
       ;;    {:_index "FOO",
       ;;     :_type "FAKE",
       ;;     :_id "NO",
       ;;     :status 400,
       ;;     :error
       ;;     {:type "invalid_index_name_exception",
       ;;      :reason "Invalid index name [FOO], must be lowercase",
       ;;      :index "FOO"}}}]}
       (let [collate-errors (fn [orig-req resp]
                              (let [item (-> resp vals first)]
                                (when (:error item)
                                  (assoc item :original-item orig-req))))
             actions-with-errors (remove nil?
                                         (map collate-errors
                                              (:body req) (:items res)))]
         (throw+ {:type ::bulk-error
                  :count (clojure.core/count actions-with-errors)
                  :items actions-with-errors})))
     res)))

(defn msearch
  ([conn req]
   (msearch conn nil req))
  ([conn idx req]
   (let [res (request conn :get (format-uri idx "_msearch") req)]
     (when (:errors res)

       (let [has-error? (fn [action]
                          (:error ((comp first vals) action)))
             actions-with-errors (filter has-error? (:items res))]
         (throw+ {:type ::msearch-error
                  :count (clojure.core/count actions-with-errors)
                  :items actions-with-errors})))
     res)))

;; Begin generated code. Everything above will be kept and anything below will be regenerated.

(defmulti clear-scroll (fn [_ & args] (map? (last args))))

(defmethod
  clear-scroll
  true
  ([conn req] (clear-scroll conn nil req))
  ([conn scroll-id req]
   (request conn :delete (format-uri "_search" "scroll" scroll-id) req)))

(defmethod
  clear-scroll
  false
  ([conn] (clear-scroll conn nil))
  ([conn scroll-id]
   (request conn :delete (format-uri "_search" "scroll" scroll-id) {})))

(defn delete
  ([conn index id] (delete conn index id nil))
  ([conn index id req]
   (request conn :delete (format-uri index id) req)))

(defn delete-by-query
  [conn index req]
  (request
   conn
   :post
   (format-uri index "_delete_by_query")
   req))

(defn delete-script
  ([conn id lang] (delete-script conn id lang nil))
  ([conn id lang req]
   (request conn :delete (format-uri "_scripts" lang id) req)))

(defn exists-source
  ([conn index id] (exists-source conn index id nil))
  ([conn index id req]
   (request conn :head (format-uri index id "_source") req)))

(defmulti field-caps (fn [_ & args] (map? (last args))))

(defmethod
  field-caps
  true
  ([conn req] (field-caps conn nil req))
  ([conn index req]
   (request conn :post (format-uri index "_field_caps") req)))

(defmethod
  field-caps
  false
  ([conn] (field-caps conn nil))
  ([conn index]
   (request conn :post (format-uri index "_field_caps") {})))

(defn get
  ([conn index id] (get conn index id nil))
  ([conn index id req]
   (request conn :get (format-uri index "_doc" id) req)))

(defn get-script
  ([conn id lang] (get-script conn id lang nil))
  ([conn id lang req]
   (request conn :get (format-uri "_scripts" lang id) req)))

(defn get-source
  ([conn index id] (get-source conn index id nil))
  ([conn index id req]
   (request conn :get (format-uri index id "_source") req)))

(defn mget
  ([conn req] (mget conn nil req))
  ([conn index req] (request conn :post (format-uri index "_mget") req)))

(defn reindex
  ([conn req] (request conn :post (format-uri "_reindex") req)))

(defn create
  ([conn index id req]
   (request conn :put (format-uri index id "_create") req)))

(defn msearch-template
  ([conn req] (msearch-template conn nil req))
  ([conn index req]
   (request
    conn
    :post
    (format-uri index "_msearch" "template")
    req)))

(defmulti mtermvectors (fn [_ & args] (map? (last args))))

(defmethod
  mtermvectors
  true
  ([conn req] (mtermvectors conn nil req))
  ([conn index req]
   (request conn :post (format-uri index "_mtermvectors") req)))

(defmethod
  mtermvectors
  false
  ([conn] (mtermvectors conn nil))
  ([conn index]
   (request conn :post (format-uri index "_mtermvectors") {})))

(defn suggest
  ([conn req] (suggest conn nil req))
  ([conn index req]
   (request conn :post (format-uri index "_suggest") req)))

(defmulti count (fn [_ & args] (map? (last args))))

(defmethod
  count
  true
  ([conn req] (count conn nil req))
  ([conn index req]
   (request conn :post (format-uri index "_count") req)))

(defmethod
  count
  false
  ([conn] (count conn nil))
  ([conn index]
   (request conn :post (format-uri index "_count") {})))

(defn put-script
  ([conn id lang req]
   (request conn :put (format-uri "_scripts" lang id) req)))

(defmulti reindex-rethrottle (fn [_ & args] (map? (last args))))

(defmethod
  reindex-rethrottle
  true
  ([conn req] (reindex-rethrottle conn nil req))
  ([conn task-id req]
   (request
    conn
    :post
    (format-uri "_update_by_query" task-id "_rethrottle")
    req)))

(defmethod
  reindex-rethrottle
  false
  ([conn] (reindex-rethrottle conn nil))
  ([conn task-id]
   (request
    conn
    :post
    (format-uri "_update_by_query" task-id "_rethrottle")
    {})))

(defmulti render-search-template (fn [_ & args] (map? (last args))))

(defmethod
  render-search-template
  true
  ([conn req] (render-search-template conn nil req))
  ([conn id req]
   (request conn :post (format-uri "_render" "template" id) req)))

(defmethod
  render-search-template
  false
  ([conn] (render-search-template conn nil))
  ([conn id]
   (request conn :post (format-uri "_render" "template" id) {})))

(defmulti search (fn [_ & args] (map? (last args))))

(defmethod
  search
  true
  ([conn req] (search conn nil req))
  ([conn index req]
   (request conn :post (format-uri index "_search") req)))

(defmethod
  search
  false
  ([conn] (search conn nil))
  ([conn index]
   (request conn :post (format-uri index "_search") {})))

(defmulti search-shards (fn [_ & args] (map? (last args))))

(defmethod
  search-shards
  true
  ([conn req] (search-shards conn nil req))
  ([conn index req]
   (request conn :post (format-uri index "_search_shards") req)))

(defmethod
  search-shards
  false
  ([conn] (search-shards conn nil))
  ([conn index]
   (request conn :post (format-uri index "_search_shards") {})))

(defmulti search-template (fn [_ & args] (map? (last args))))

(defmethod
  search-template
  true
  ([conn req] (search-template conn nil req))
  ([conn index req]
   (request
    conn
    :post
    (format-uri index "_search" "template")
    req)))

(defmethod
  search-template
  false
  ([conn] (search-template conn nil))
  ([conn index]
   (request conn :post (format-uri index "_search" "template") {})))

(defmulti termvectors (fn [_ & args] (map? (last args))))

(defmethod
  termvectors
  true
  ([conn index req] (termvectors conn index nil req))
  ([conn index id req]
   (request conn :post (format-uri index id "_termvectors") req)))

(defmethod
  termvectors
  false
  ([conn index] (termvectors conn index nil))
  ([conn index id]
   (request conn :post (format-uri index id "_termvectors") {})))

(defn update
  ([conn index id] (update conn index id nil))
  ([conn index id req]
   (request conn :post (format-uri index id "_update") req)))

(defn explain
  ([conn index id] (explain conn index id nil))
  ([conn index id req]
   (request conn :post (format-uri index id "_explain") req)))

(defmulti count-percolate (fn [_ & args] (map? (last args))))

(defmethod
  count-percolate
  true
  ([conn index req] (count-percolate conn index nil req))
  ([conn index id req]
   (request
    conn
    :post
    (format-uri index id "_percolate" "count")
    req)))

(defmethod
  count-percolate
  false
  ([conn index] (count-percolate conn index nil))
  ([conn index id]
   (request
    conn
    :post
    (format-uri index id "_percolate" "count")
    {})))

(defn delete-template
  ([conn id] (delete-template conn id nil))
  ([conn id req]
   (request conn :delete (format-uri "_search" "template" id) req)))

(defmulti field-stats (fn [_ & args] (map? (last args))))

(defmethod
  field-stats
  true
  ([conn req] (field-stats conn nil req))
  ([conn index req]
   (request conn :post (format-uri index "_field_stats") req)))

(defmethod
  field-stats
  false
  ([conn] (field-stats conn nil))
  ([conn index]
   (request conn :post (format-uri index "_field_stats") {})))

(defn get-template
  ([conn id] (get-template conn id nil))
  ([conn id req]
   (request conn :get (format-uri "_search" "template" id) req)))

(defmulti percolate (fn [_ & args] (map? (last args))))

(defmethod
  percolate
  true
  ([conn index req] (percolate conn index nil req))
  ([conn index id req]
   (request conn :post (format-uri index id "_percolate") req)))

(defmethod
  percolate
  false
  ([conn index] (percolate conn index nil))
  ([conn index id]
   (request conn :post (format-uri index id "_percolate") {})))

(defn put-template
  ([conn id req]
   (request conn :put (format-uri "_search" "template" id) req)))

(defmulti update-by-query (fn [_ & args] (map? (last args))))

(defmethod
  update-by-query
  true
  [conn index req]
  (request conn :post (format-uri index "_update_by_query") req))

(defmethod
  update-by-query
  false
  [conn index]
  (request conn :post (format-uri index "_update_by_query") {}))
