(ns elasticsearch.codegen
  (:gen-class)
  (:require [clojure.java.io :as io]
            [clojure.pprint :refer [pprint]]
            [clojure.set :refer [subset?]]
            [clojure.string :as str]
            [clojure.walk :refer [keywordize-keys]]
            [elasticsearch.common :refer [format-uri request]]
            [elasticsearch.json :as json]))

;; Code generation logic

(def req (symbol "req"))

(def conn (symbol "conn"))

(defn move-id-arg
  "There are a few APIs that require an index, type, and id to be passed. The
  REST API specs for these functions typically order them as [:id :index :type],
  but URL paths are universally ordered as /{index}/{type}/{id}. If we encounter
  an argument vector that contains these three, this function will order them in
  the latter fashion by 'moving' :id to the end of the collection.."
  [args]
  (if (subset? #{:id :index :type} (set args))
    (let [offset (count (take-while #(not= :id %) args))
          end (count args)]
      (if (= [:id :index :type] (subvec args offset end))
        (concat (take offset args) (subvec args (inc offset) end) [:id])
        args))
    args))

(defn req-last
  "No matter what, we always want the req argument to be the last
  argument in an argument vector."
  [args]
  (if (some #{req} args)
    (vec (concat (remove #(= req %) args) [req]))
    args))

(defn underscores-are-bad-and-should-feel-bad [x]
  (str/replace (name x) #"_" "-"))

(defn transform-args
  "Transforms a vec of keywords (possibly containing underscores) into vec of
  propertly-formatted symbols with the id symbol in the correct position."
  [args]
  (->> args
       move-id-arg
       (map (comp symbol underscores-are-bad-and-should-feel-bad))))

(defn arg-vectors
  "Takes a collection of required arguments and a collection of optional arguments.
  Returns a collection containing all valid combinatons of the two. For example,

  (arg-vectors [:conn :index :type] [:id :req]) will return:

  ([conn index type] [conn index type id] [conn index type id req])"
  [required optional]
  (map req-last
       (reduce (fn [valid-combinations args]
                 (let [form (into [] (concat
                                      (transform-args required)
                                      args))]
                   (conj valid-combinations form)))
               []
               (reductions conj [] (transform-args optional)))))

(defn replace-sym [s]
  (if (.startsWith s "{")
    (if (.endsWith s "}")
      (symbol (underscores-are-bad-and-should-feel-bad
               (subs s 1 (- (count s) 1))))
      (throw (Exception. (str "Malformed variable in REST spec: " s))))
    s))

(defn format-path [path]
  (->> (str/split path #"/")
       (remove empty?)
       (map replace-sym)))

(defn request-call
  "Takes a method, a path, and req? (a boolean.) Returns an appropriate call to
  elasticsearch.conn/request with the method, path, and either the symbol req or
  an empty map, as elasticsearch.conn/request expects a map as its last
  argument. For a variadic function, this will serve as the base implementation
  that all other arities ultimately call."
  [method path req?]
  `(~'request ~conn
              ~method
              (~'format-uri ~@(format-path path))
              ~(if req? req {})))

(defn largest-arity [impl arg-vecs]
  (list (last arg-vecs) impl))

(defn last-arg-req? [arg-vec]
  (some #(= req %) arg-vec))

(defn recursive-call
  "Returns an implementation of a variadic function that calls the next highest
  arity of itself with an additional value as its last argument. This last value
  will either be nil or req, depending on whether req is the last argument in
  the given arg-vec."
  [fn-name arg-vec]
  (let [args (if (last-arg-req? arg-vec)
               (concat (butlast arg-vec) [nil req])
               (concat arg-vec [nil]))
        invocation `(~(symbol fn-name) ~@args)]
    `(~arg-vec ~invocation)))

(defn gen-forms
  "The core of the code generation logic. Returns the forms that will ultimately
  make up the body of a variadic function."
  [fn-name method path required-args optional-args req?]
  (let [args (arg-vectors required-args optional-args)
        base-impl (largest-arity (request-call method path (or req? false)) args)
        small-arities (map (partial recursive-call fn-name) (butlast args))]
    (if (>= (count small-arities) 1)
      (concat small-arities (list base-impl))
      (list base-impl))))

(defn defn-form [fn-name method path required-args optional-args]
  (let [forms (gen-forms fn-name method path required-args optional-args true)]
    `(~'defn ~(symbol fn-name)
       ~@forms)))

(defn defmulti-form [fn-name]
  `(~'defmulti ~(symbol fn-name)
     (~'fn [~'_ & ~'args] (~'map? (~'last ~'args)))))

(defn defmethod-form [fn-name method path required-args optional-args dispatch]
  (let [forms (gen-forms fn-name method path required-args optional-args dispatch)]
    `(~'defmethod ~(symbol fn-name) ~(symbol (str dispatch))
       ~@forms)))

(defn defmethod-form-true [fn-name method path required-args optional-args]
  (defmethod-form fn-name method path required-args optional-args true))

(defn defmethod-form-false [fn-name method path required-args optional-args]
  (defmethod-form fn-name method path required-args optional-args false))

;; REST API spec parsing
(defn original-function-name [spec]
  (-> spec
      keys
      first))

(defn function-name [spec]
  (let [parts (-> (original-function-name spec)
                  name
                  (str/split #"[\\.]"))
        placer (if (> (count parts) 1) second first)]
    (-> parts
        placer
        underscores-are-bad-and-should-feel-bad)))

(defn method
  "Certain Elasticsearch APIs consider both POST and GET to be valid HTTP
  methods. For example, searches can be initiated with either a POST or a GET.
  Certain proxies do not consider GET requests with bodies to be valid. So, to
  account for this, we'll default to POST as the method for an API that accepts
  both."
  [spec]
  (let [methods (:methods (first (vals spec)))]
    (if (= methods ["GET" "POST"])
      :post
      (keyword (str/lower-case (first methods))))))

(defn url-path
  [spec]
  (-> spec
      vals
      first
      :url
      :paths
      sort
      last))

(defn body-required? [spec fn-name]
  (-> spec
      (get (keyword fn-name))
      :body
      :required))

(defn default-args [spec orig-fn-name impl]
  (condp = impl
    #'defn-form (if (body-required? spec orig-fn-name)
                  [[:conn :req] []]
                  [[:conn] [:req]])
    #'defmethod-form-true [[:conn :req] []]
    #'defmethod-form-false [[:conn] []]))

(defn fn-args [spec & [impl]]
  (let [orig-fn-name (original-function-name spec)
        [required optional] (if impl
                              (default-args spec orig-fn-name impl)
                              [[:conn] []])
        all-args (-> spec
                     vals
                     first
                     :url
                     :parts)]
    (reduce-kv (fn [acc k v]
                 (if (:required v)
                   (update acc :required #(conj % k))
                   (update acc :optional #(conj % k))))
               {:required required
                :optional optional}
               all-args)))

;; Action

(def spec-dir
  "../elasticsearch/rest-api-spec/src/main/resources/rest-api-spec/api/")

;; We'll maintain hand-coded implementations of these functions
(def blacklist
  {:cluster #{"stats"}
   :document #{"bulk"
               "exists"
               "index"
               "info"
               "mpercolate"
               "msearch"
               "ping"
               "scroll"}
   :indices #{"create"
              "exists"
              "exists_alias"
              "exists_template"
              "exists_type"
              "get_field_mapping"
              "put_mapping"}})

(defn blacklisted? [blacklisted-fns path]
  (let [parts (-> path
                  .getName
                  (str/split #"[\\.]"))
        f (if (= 2 (count parts))
            (first parts)
            (second parts))]
    (contains? blacklisted-fns f)))

(defn document-specs
  "The elasticsearch.document namespace is a catch-all for APIs that aren't
  namespaced by Elasticsearch itself."
  []
  (let [other-ns #"cat|cluster|indices|ingest|nodes|snapshot|tasks"
        blacklisted-fns (get blacklist :document)
        to-remove (re-pattern (str/join "|" blacklisted-fns))]
    (->> (file-seq (io/file spec-dir))
         (remove #(.isDirectory %))
         (remove #(re-find other-ns (str %)))
         (remove (partial blacklisted? blacklisted-fns)))))

(defn specs-for-namespace [namespace]
  (if (= "document" namespace)
    (document-specs)
    (let [blacklisted-fns (get blacklist (keyword namespace))
          specs (->> (file-seq (io/file spec-dir))
                     (filter #(re-find (re-pattern (str "api/" namespace)) (str %))))]
      (if (empty? blacklisted-fns)
        specs
        (remove (partial blacklisted? blacklisted-fns) specs)))))

(defn gen-fn-base [spec fn-name impl]
  (let [meth (method spec)
        path (url-path spec)
        args (fn-args spec impl)]
    (impl fn-name meth path (:required args) (:optional args))))

(defmulti gen-fn
  "The dispatch function's core logic is as follows:

  If the function requires a body, there's no point in using multimethods, as
  regular defn forms can express this easily.

  If there are fewer than three forms, there's also no point in using
  multimethods.

  In any other case, we need to use multimethods to express the largest number
  of possible valid combinations of arguments."
  (fn [spec fn-name]
    (let [meth (method spec)
          path (url-path spec)
          args (fn-args spec)
          forms (if (body-required? spec (original-function-name spec))
                  (gen-forms fn-name meth path
                             (conj (:required args) :req) (:optional args) true)
                  (gen-forms fn-name meth path
                             (:required args) (conj (:optional args) :req) true))]
      (cond
        (body-required? spec (original-function-name spec)) :defn
        (< (count forms) 3) :defn
        :else :multimethod))))

(defmethod gen-fn :multimethod [spec fn-name]
  (with-out-str
    (pprint (defmulti-form fn-name))
    (println)
    (pprint (gen-fn-base spec fn-name #'defmethod-form-true))
    (println)
    (pprint (gen-fn-base spec fn-name #'defmethod-form-false))))

(defmethod gen-fn :defn [spec fn-name]
  (with-out-str
    (clojure.pprint/write (gen-fn-base spec fn-name #'defn-form)
                          :dispatch clojure.pprint/code-dispatch)
    (println)))

(def code-separator
  (str ";; Begin generated code. Everything above will be kept"
       " and anything below will be regenerated."))

(defn forms-to-keep
  "In order to avoid overwriting blacklisted function implementations, and to
  allow us to keep extra-API function definitions in the same files as the
  generated code, we need a way to pass over these forms in a given file. This
  function returns a string containing all such forms. We use a comment string
  to separate hand-maintained code from generated code."
  [namespace]
  (let [path (str "src/elasticsearch/" namespace ".clj")]
    (->> path
         slurp
         str/split-lines
         (take-while #(not= code-separator %))
         (str/join "\n"))))

(defn new-forms [namespace]
  (with-out-str
    (println)
    (doseq [f (specs-for-namespace namespace)
            :let [spec (keywordize-keys (json/decode (slurp f)))
                  fn-name (function-name spec)]]
      (println (gen-fn spec fn-name)))))

(defn write-ns [namespace]
  (let [to-keep (forms-to-keep namespace)
        updated (new-forms namespace)
        str-to-write (str/join "\n" [to-keep code-separator updated])]
    (spit (str "src/elasticsearch/" namespace ".clj") str-to-write)))

(defn namespaces []
  (->> spec-dir
       io/file
       file-seq
       rest
       (map #(.getName %))
       (map #(str/split % #"[\\.]"))
       (filter #(> (count %) 2))
       (map first)
       set))

(defn -main [& args]
  (doseq [namespace (conj (namespaces) "document")]
    (println "Updating namespace: " namespace)
    (write-ns namespace)))
