(ns elasticsearch.async
  (:require [cheshire.core :as json]
            [clojure.core.async
             :refer [chan go <! <!! >! >!! alts! alts!!
                     close!]
             :as async]))

(defn buffering-proc
  "Takes a function f and returns an input and output channel.  Items
  put into the input channel will be buffered together until
  either (pred buf), or interval millis is exhausted, at which
  point (f buf) will be applied if (has-items? buf).

  The return value of each call to f will be put into the output
  channel."
  ([f new-buf conj-item has-items? get-items preds interval
    queue-size-in queue-size-out]
   (let [in (if (>= queue-size-in 0)
              (chan queue-size-in)
              (chan))
         out (if (>= queue-size-out 0)
               (chan queue-size-out)
               (chan))]
     (go
       (loop [buf (new-buf)]
         (let [[x ch] (alts! [in (async/timeout interval)])]
           (conj-item buf x)
           (cond
             ;; interval expired, flush and reset buffer
             (nil? x)
             (do
               (when (has-items? buf)
                 (>! out (f (get-items buf))))
               (recur (new-buf)))

             ;; at least one pred true, reset buffer
             (some identity (for [pred preds] (pred buf)))
             (do
               (when (has-items? buf)
                 (>! out (f (get-items buf))))
               (recur (new-buf)))

             ;; nothing to do, go again
             :else
             (recur buf)))))
     [in out])))

(defn bulk-buffering-proc
  ([f max-bytes max-actions interval queue-size-req queue-size-resp]
   (let [buf-init (fn []
                    (atom {:xs [] :bytes 0}))
         count+ (fn [v x]
                  (if x
                    (-> v
                        (update-in [:xs] conj x)
                        (update-in [:bytes] + (count (json/encode x))))
                    v))]
     (buffering-proc
      f
      buf-init
      (fn [buf val] (swap! buf count+ val))
      (fn [buf] (pos? (count (:xs @buf))))
      (fn [buf] (:xs @buf))
      [(fn [buf] (and max-bytes (>= (:bytes @buf) max-bytes)))
       (fn [buf] (and max-actions (>= (count (:xs @buf)) max-actions)))]
      interval
      queue-size-req
      queue-size-resp))))
