(ns elasticsearch.indices
  (:refer-clojure :exclude [type count get])
  (:require [clojure.string :as str]
            [elasticsearch.common :refer [fmt-uri request]]
            [elasticsearch.connection :as conn]
            [slingshot.slingshot :refer [try+]]))

(defn create [conn {:keys [index] :as req}]
  (request conn :put (fmt-uri index) req))

(defn delete [conn {:keys [index] :as req}]
  (request conn :delete (fmt-uri index) req))

(defn get-settings [conn {:keys [index name] :as req}]
  (let [parts [(if index index "_all") "_settings" name]]
    (request conn :get (apply fmt-uri parts) req)))

(defn get-mapping [conn {:keys [index type] :as req}]
  (let [parts [(if index index "_all") "_mapping" type]]
    (request conn :get (apply fmt-uri parts) req)))

(defn put-mapping [conn {:keys [index type] :as req}]
  (request conn :put (fmt-uri index "_mapping" type) req))

(defn health [conn {:keys [index] :as req}]
  (request conn :get (fmt-uri "_cluster" "health" index) req))

(defn wait-for-health [conn status {:keys [index] :as req}]
  (health conn (merge req {:wait_for_status (name status)})))

(defn refresh [conn {:keys [index] :as req}]
  (request conn :post (apply fmt-uri [index "_refresh"]) req))

(defn count [conn {:keys [index] :as req}]
  (request conn :get (fmt-uri index "_count") req))

(defn analyze [conn {:keys [index] :as req}]
  (request conn :get (fmt-uri index "_analyze") req))

(defn clear-cache [conn {:keys [index] :as req}]
  (request conn :post (fmt-uri index "_cache" "clear") req))

(defn close [conn {:keys [index] :as req}]
  (request conn :post (fmt-uri index "_close") req))

(defn open [conn {:keys [index] :as req}]
  (request conn :post (fmt-uri index "_open") req))

(defn delete-alias  [conn {:keys [index name] :as req}]
  (request conn :delete (fmt-uri index "_alias" name) req))

(defn delete-template [conn {:keys [name] :as req}]
  (request conn :delete (fmt-uri "_template" name) req))

(defn exists [conn {:keys [index type id] :as req}]
  (try+
   (request conn :head (fmt-uri index type id) req)
   true
   (catch [:status 404] _
     false)))

(defn exists-alias [conn {:keys [index name] :as req}]
  (try+
   (request conn :head (fmt-uri index "_alias" name) req)
   true
   (catch [:status 404] _
     false)))

(defn exists-type [conn {:keys [index type] :as req}]
  (try+
   (request conn :head (fmt-uri index type) req)
   true
   (catch [:status 404] _
     false)))

(defn flush-synced [conn {:keys [index] :as req}]
  (request conn :post (fmt-uri index "_flush" "synced") req))

(defn forcemerge [conn {:keys [index] :as req}])

(defn get-alias [conn {:keys [index name] :as req}]
  (request conn :get (fmt-uri index "_alias" name) req))

(defn get-field-mapping [conn {:keys [index type fields] :as req}]
  (let [parts [(if index index "_all")
               type
               "_mapping"
               "field"
               fields]]
    (request conn :get (apply fmt-uri parts) req)))

(defn put-alias [conn {:keys [index name] :as req}]
  (request conn :post (fmt-uri index "_alias" name) req))

(defn put-settings [conn {:keys [index] :as req}]
  (request conn :put (fmt-uri index "_settings") req))

(defn get-template [conn {:keys [name] :as req}]
  (request conn :get (fmt-uri "_template" name) req))

(defn exists-template [conn {:keys [name] :as req}]
  (try+
   (request conn :head (fmt-uri "_template" name) req)
   true
   (catch [:status 404] _
     false)))

(defn put-template [conn {:keys [name] :as req}]
  (request conn :put (fmt-uri "_template" name) req))

(defn recovery [conn {:keys [index] :as req}]
  (request conn :get (fmt-uri index "_recovery") req))

(defn rollover [conn {:keys [alias index] :as req}]
  (request conn :post (fmt-uri alias "_rollover" index) req))

(defn segments [conn {:keys [index] :as req}]
  (request conn :get (fmt-uri index "_segments") req))

(defn get [conn {:keys [index feature] :as req}]
  (request conn :get (fmt-uri index feature) req))

(defn shard-stores [conn {:keys [index] :as req}]
  (request conn :get (fmt-uri index "_shard_stores") req))

(defn stats [conn {:keys [index] :as req}]
  (request conn :get (fmt-uri index "_stats") req))

(defn update-aliases [conn {:keys [index name] :as req}]
  (request conn :post (fmt-uri index "_aliases" name) req))

(defn upgrade [conn {:keys [index] :as req}]
  (request conn :post (fmt-uri index "_upgrade") req))

(defn validate-query [conn {:keys [index] :as req}]
  (request conn :get (fmt-uri index "_validate" "query") req))

(defn shrink [conn {:keys [index target] :as req}]
  (request conn :post (fmt-uri index "_shrink" target) req))

(defmacro with-tmp-index [conn sym settings & body]
  `(let [~sym (format "tmp-%s" (->> (java.util.UUID/randomUUID)
                                    str (take 6) (apply str)))]
     (try
       (create ~conn {:index ~sym :body ~settings})
       (wait-for-health ~conn :yellow {:index ~sym})
       ~@body
       (finally
         (delete ~conn {:index ~sym})))))
