(ns elasticsearch.document
  (:refer-clojure :exclude [get type update count])
  (:require [clojure.string :as str]
            [elasticsearch.common :refer [fmt-uri request]]
            [elasticsearch.connection :as conn]
            [elasticsearch.connection.http :as hconn]
            [elasticsearch.schema :refer :all]
            [slingshot.slingshot :refer [throw+]]))

(defn index [conn {:keys [index type id] :as req}]
  (let [method (if id
                 :put
                 :post)]
    (request conn method (fmt-uri index type id) req)))

(defn create [conn {:keys [index type id] :as  req}]
  (#'index conn (merge req {:op_type "create"})))

(defn get [conn {:keys [index type id] :as req}]
  (request conn :get (fmt-uri index type id) req))

(defn delete [conn {:keys [index type id] :as req}]
  (request conn :delete (fmt-uri index type id) req))

(defn get-source [conn {:keys [index type id] :as  req}]
  (request conn :get (fmt-uri index type id "_source") req))

(defn update [conn {:keys [index type id] :as req}]
  (request conn :post (fmt-uri index type id "_update") req))

(defn exists?
  [conn {:keys [index type id] :as req}]
  (let [req (assoc-in req [:conn-params :raw-response?] true)]
    (= 200
       (:status
        (request conn :head (fmt-uri index type id) req)))))

(defn search [conn {:keys [index type id] :as req}]
  (request conn :post (fmt-uri index type id "_search") req))

(defn count [conn {:keys [index type] :as req}]
  (request conn :post (fmt-uri index type "_count") req))

(defn bulk
  [conn {:keys [index type] :as req}]
  (let [res (request conn :post (fmt-uri index type "_bulk") req)]
    (when (:errors res)
      ;; A response that contains an error looks like:
      ;;
      ;; {:took 17,
      ;;  :errors true,
      ;;  :items
      ;;  [{:delete
      ;;    {:_index "FOO",
      ;;     :_type "FAKE",
      ;;     :_id "NO",
      ;;     :status 400,
      ;;     :error
      ;;     {:type "invalid_index_name_exception",
      ;;      :reason "Invalid index name [FOO], must be lowercase",
      ;;      :index "FOO"}}}]}
      (let [has-error? (fn [action]
                         (:error ((comp first vals) action)))
            actions-with-errors (filter has-error? (:items res))]
        (throw+ {:type ::bulk-error
                 :count (clojure.core/count actions-with-errors)
                 :items actions-with-errors})))
    res))

(defn suggest [conn req]
  (request conn :post (fmt-uri "_suggest") req))

(defn termvectors [conn {:keys [index type id] :as req}]
  (request conn :get (fmt-uri index type id "_termvectors") req))

(defn explain [conn {:keys [index type id] :as req}]
  (request conn :get (fmt-uri index type id "_explain") req))

(defn mget [conn {:keys [index type] :as req}]
  (request conn :get (fmt-uri index type "_mget") req))

(defn field-stats [conn {:keys [index] :as req}]
  (request conn :get (fmt-uri index "_field_stats") req))

(defn search-shards [conn & [req]]
  (request conn :get (fmt-uri "_search_shards") req))

(defn mget [conn {:keys [index type] :as req}]
  (request conn :get (fmt-uri index type "_mget") req))

(defn msearch
  [conn {:keys [index type] :as req}]
  (let [res (request conn :get (fmt-uri index type "_msearch") req)]
    (when (:errors res)

      (let [has-error? (fn [action]
                         (:error ((comp first vals) action)))
            actions-with-errors (filter has-error? (:items res))]
        (throw+ {:type ::msearch-error
                 :count (clojure.core/count actions-with-errors)
                 :items actions-with-errors})))
    res))

(defn mtermvectors [conn {:keys [index type] :as req}]
  (request conn :get (fmt-uri index type "_mtermvectors") req))
