(ns elasticsearch.connection.http
  (:refer-clojure :exclude [get update])
  (:require [clj-http.client :as http]
            [clj-http.core]
            clojure.pprint
            [clojure.string :as string]
            [elasticsearch.connection :as conn]
            [elasticsearch.json :as json]
            [schema.core :as s]
            [slingshot.slingshot :refer [try+ throw+]])
  (:import (elasticsearch.connection Connection)))

(defn bulk-assemble [action action-name doc]
  (let [parts [(update-in action [action-name] dissoc :source)
               (:source doc)]]
    (string/join
     "\n"
     (map json/encode (filter identity parts)))))

(defn bulk-stringify [act]
  (let [k (first (keys act))
        doc (act k)]
    (condp = k
      :index (bulk-assemble act k doc)
      :create (bulk-assemble act k doc)
      :update (bulk-assemble act k doc)
      ;; no payload here
      :delete (json/encode act)
      ;; Default
      (json/encode act))))

(defn make-bulk-body [actions]
  (str
   (->> actions
        (map bulk-stringify)
        (interpose "\n")
        (filter identity)
        (apply str))
   "\n"))

(defn make-msearch-body [queries]
  (str
   (->> queries
        (map json/encode)
        (interpose "\n")
        (apply str))
   "\n"))

(defmulti coerce-request-body
  (fn [req]
    (cond
      (.endsWith (:uri req) "_bulk") :bulk
      (.endsWith (:uri req) "_msearch") :msearch
      :else :default)))

(defmethod coerce-request-body :bulk
  [req]
  (if (:body req)
    (assoc req :body (make-bulk-body (:body req)))
    req))

(defmethod coerce-request-body :msearch
  [req]
  (if (:body req)
    (assoc req :body (make-msearch-body (:body req)))
    req))

(defmethod coerce-request-body :default
  [req]
  (if (:body req)
    (assoc req :body (json/encode (:body req)))
    req))

(defn wrap-coerce-request-body
  [client]
  (fn [req]
    (client (coerce-request-body req))))

(defn wrap-coerce-response-body
  "Automatically sends {:as :json} for all requests except for /_cat API, which
  sends {:as :string}."
  [client]
  (fn [req]
    (let [req-type (if (.contains (:uri req) "_cat") :string :json)
          req (merge {:as req-type} req)]
      (client req))))

(defn text-response? [res]
  (let [content-type (clojure.core/get (:headers res) "Content-Type")]
    (.contains content-type "text/plain")))

(defn wrap-exception-handling
  "Decode the body in JSON exceptions and re-throw the exception"
  [client]
  (fn [req]
    (try+
     (let [res (client req)]
       (if (and (map? res)
                (string? (:body res))
                (not (text-response? res)))
         ;; an exception happened with {:throw-exceptions? false}
         (assoc res :body (json/decode (:body res)))
         ;; normal response with no errors
         res))
     (catch map? m
       (let [err-body (if (and (string? (:body m))
                               (not (text-response? m)))
                        (json/decode (:body m))
                        (:body m))]
         ;; an exception happened with {:throw-exceptions? true}
         (throw+ (assoc m :body err-body)))))))

(defn wrap-ignore-exception
  [client]
  (fn [req]
    (if-let [err-code (:ignore req)]
      (let [err-code (if (coll? err-code)
                       (first err-code)
                       err-code)]
        (try+
         (client req)
         (catch [:status err-code] _)))
      (client req))))

(defn wrap-raw-response
  "Whether to return the :body as the payload or the whole response"
  [client]
  (fn [req]
    (let [res (client req)]
      (if (:raw-response? req)
        res
        (:body res)))))

(defn wrap-conn-params
  "Check for last-minute settings for single requests"
  [client]
  (fn [req]
    (client
     (if (:conn-params req)
       (merge
        (dissoc req :conn-params)
        (:conn-params req))
       req))))

(defn wrap-debug-request [client]
  (fn [req]
    (clojure.pprint/pprint req)
    (client req)))

(def elasticsearch-middleware
  [;;wrap-debug-request

   clj-http.client/wrap-request-timing
   clj-http.headers/wrap-header-map
   clj-http.client/wrap-query-params
   clj-http.client/wrap-basic-auth
   clj-http.client/wrap-oauth
   clj-http.client/wrap-user-info

   wrap-coerce-request-body

   clj-http.client/wrap-redirects
   clj-http.client/wrap-decompression
   clj-http.client/wrap-input-coercion
   ;;clj-http.client/wrap-additional-header-parsing
   clj-http.client/wrap-output-coercion

   wrap-coerce-response-body

   clj-http.client/wrap-exceptions
   clj-http.client/wrap-accept
   clj-http.client/wrap-accept-encoding
   clj-http.client/wrap-content-type
   ;;clj-http.client/wrap-form-params
   clj-http.client/wrap-nested-params
   clj-http.client/wrap-method
   ;; Do we really need cookies? TODO check if shield needs theme
   clj-http.cookies/wrap-cookies
   clj-http.links/wrap-links
   clj-http.client/wrap-unknown-host

   wrap-ignore-exception
   wrap-exception-handling
   wrap-raw-response
   wrap-conn-params])

(defn wrap-request
  [request]
  (reduce (fn [request middleware]
            (middleware request))
          request
          elasticsearch-middleware))

(def do-request
  (wrap-request #'clj-http.core/request))

(defrecord HttpConnection [settings]
  Connection
  (request [this method opts]
    (assert (not (nil? method)))
    (do-request
     (merge
      (:settings this)
      {:method method}
      opts))))

(s/defn make :- HttpConnection
  [settings :- {:url s/Str
                s/Keyword s/Any}]
  (->HttpConnection
   (-> settings (dissoc :url) (merge (http/parse-url (:url settings))))))
