(ns elasticsearch.document
  (:refer-clojure :exclude [get type update])
  (:require [elasticsearch.connection :as conn]
            [elasticsearch.schema :refer :all]
            [schema.core :as s]
            [slingshot.slingshot :refer [throw+]])
  (:import
   ;; For schema, need the class
   (elasticsearch.connection Connection)))

(defn make-uri [{:keys [index type id action]}]
  (->> [index type id action]
       (filter identity)
       (map #(java.net.URLEncoder/encode %))
       (clojure.string/join "/")
       (format "/%s")))

(defn request
  [conn method {:keys [index type id body] :as req} & [opts]]
  (conn/request conn method (merge-with merge
                                        (dissoc req :index :type :id)
                                        {:uri (make-uri req)}
                                        opts)))

(s/defn index
  "Index a document into cluster/index/type/id"
  [conn :- Connection
   req :- IndexRequest]
  ;; use POST since it will handle absent and present _id
  (request conn :post req))

(s/defn create
  "Create a document into cluster/index/type/id, will fail if document
  already exists"
  [conn :- Connection
   req :- CreateRequest]
  (request conn :put (assoc req :action "_create")))

(s/defn get
  "GET a document from a cluster/index/type/id"
  [conn :- Connection
   req :- GetRequest]
  (request conn :get req))

(s/defn delete
  "DELETE a document from a cluster/index/type/id"
  [conn :- Connection
   req :- DeleteRequest]
  (request conn :delete req))

(s/defn get-source
  "GET a document, source-only"
  [conn :- Connection
   req :- GetRequest]
  (request conn :get (assoc req :action "_source")))

(s/defn update
  "Update a document"
  [conn :- Connection
   req :- UpdateRequest]
  (request conn :post (assoc req :action "_update")))

(s/defn exists? :- s/Bool
  "Check if a doc exists already"
  [conn :- Connection
   req :- DocExistsRequest]
  (= 200
     (:status
      (request conn :head (assoc-in req [:conn-params :raw-response?] true)))))

(s/defn search
  [conn :- Connection
   req :- SearchRequest]
  (request conn :post (assoc req :action "_search")))

(s/defn bulk
  [conn :- Connection
   req :- BulkRequest]
  (let [res (request conn :post (assoc req :action "_bulk"))]
    (when (:errors res)
      ;; A response that contains an error looks like:
      ;;
      ;; {:took 17,
      ;;  :errors true,
      ;;  :items
      ;;  [{:delete
      ;;    {:_index "FOO",
      ;;     :_type "FAKE",
      ;;     :_id "NO",
      ;;     :status 400,
      ;;     :error
      ;;     {:type "invalid_index_name_exception",
      ;;      :reason "Invalid index name [FOO], must be lowercase",
      ;;      :index "FOO"}}}]}
      (let [has-error? (fn [action]
                         (:error ((comp first vals) action)))
            actions-with-errors (filter has-error? (:items res))]
        (throw+ {:type ::bulk-error
                 :count (count actions-with-errors)
                 :items actions-with-errors})))
    res))
