(ns elasticsearch.connection.http
  (:refer-clojure :exclude [get update])
  (:require [clj-http.client :as http]
            [clj-http.core]
            [clojure.string :as string]
            [elasticsearch.connection :as conn]
            [elasticsearch.json :as json]
            [slingshot.slingshot :refer [try+ throw+]])
  (:import (elasticsearch.connection Connection)))

(defn make-bulk-body [actions]
  (let [assemble (fn [action action-name doc]
                   (string/join
                    "\n"
                    [(json/encode
                      (update-in action [action-name] dissoc :source))
                     (json/encode (:source doc))]))
        stringify
        (fn [act]
          (let [k (first (keys act))
                doc (act k)]
            (condp = k
              :index (assemble act k doc)
              :create (assemble act k doc)
              :update (assemble act k doc)
              ;; no payload here
              :delete (json/encode act)
              (throw (Exception. (format "unsupported action %s" k))))))]
    (str
     (->> actions
          (map stringify)
          (interpose "\n")
          (apply str))
     "\n")))

(defmulti coerce-request-body
  (fn [req]
    (cond
      (.endsWith (:uri req) "_bulk") :bulk
      :else :default)))

(defmethod coerce-request-body :bulk
  [req]
  (if (:body req)
    (assoc req :body (make-bulk-body (:body req)))
    req))

(defmethod coerce-request-body :default
  [req]
  (if (:body req)
    (assoc req :body (json/encode (:body req)))
    req))

(defn wrap-coerce-request-body
  [client]
  (fn [req]
    (client (coerce-request-body req))))

(defn wrap-coerce-response-body
  "Automatically sends {:as :json} for all requests except for /_cat API, which
  sends {:as :string}."
  [client]
  (fn [req]
    (let [req-type (if (.contains (:uri req) "_cat") :string :json)
          req (merge {:as req-type} req)]
      (client req))))

(defn wrap-exception-handling
  "Decode the body in JSON exceptions and re-throw the exception"
  [client]
  (fn [req]
    (try+
     (let [res (client req)]
       (if (and (map? res)
                (string? (:body res)))
         ;; an exception happened with {:throw-exceptions? false}
         (assoc res :body (json/decode (:body res)))
         ;; normal response with no errors
         res))
     (catch map? m
       (let [err-body (json/decode (:body m))]
         ;; an exception happened with {:throw-exceptions? true}
         (throw+ (assoc m :body err-body)))))))

(defn wrap-raw-response
  "Whether to return the :body as the payload or the whole response"
  [client]
  (fn [req]
    (let [res (client req)]
      (if (:raw-response? req)
        res
        (:body res)))))

(defn wrap-conn-params
  "Check for last-minute settings for single requests"
  [client]
  (fn [req]
    (client
     (if (:conn-params req)
       (merge
        (dissoc req :conn-params)
        (:conn-params req))
       req))))

(def elasticsearch-middleware
  [clj-http.client/wrap-request-timing
   clj-http.headers/wrap-header-map
   clj-http.client/wrap-query-params
   clj-http.client/wrap-basic-auth
   clj-http.client/wrap-oauth
   clj-http.client/wrap-user-info

   wrap-coerce-request-body

   clj-http.client/wrap-redirects
   clj-http.client/wrap-decompression
   clj-http.client/wrap-input-coercion
   ;;clj-http.client/wrap-additional-header-parsing
   clj-http.client/wrap-output-coercion

   wrap-coerce-response-body

   clj-http.client/wrap-exceptions
   clj-http.client/wrap-accept
   clj-http.client/wrap-accept-encoding
   clj-http.client/wrap-content-type
   ;;clj-http.client/wrap-form-params
   clj-http.client/wrap-nested-params
   clj-http.client/wrap-method
   ;; Do we really need cookies? TODO check if shield needs theme
   clj-http.cookies/wrap-cookies
   clj-http.links/wrap-links
   clj-http.client/wrap-unknown-host

   wrap-exception-handling
   wrap-raw-response
   wrap-conn-params])

(defn wrap-request
  [request]
  (-> request)
  (reduce (fn [request middleware]
            (middleware request))
          request
          elasticsearch-middleware))

(def do-request
  (wrap-request #'clj-http.core/request))

(defrecord HttpConnection [settings]
  Connection
  (request [this method opts]
    (assert (not (nil? method)))
    (do-request
     (merge
      (:settings this)
      {:method method}
      opts))))

(defn make
  [settings]
  (if-let [url (:url settings)]
    (->HttpConnection
     (-> settings (dissoc :url) (merge (http/parse-url url))))
    (->HttpConnection settings)))
