(ns logplots.histo
  "
  Adapted from my blog post, http://johnj.com/nucleotide-repetition-lengths.html
  "
  (:import [org.jfree.data.xy XYSeriesCollection XYSeries]
           [org.jfree.chart ChartFrame JFreeChart]
           [org.jfree.chart.plot XYPlot]
           [org.jfree.chart.axis NumberAxis]
           [org.jfree.chart.renderer.xy XYBarRenderer StandardXYBarPainter]))

(defn make-hist
  "
  Convert seq of input xs into a histogram of nbins bins, from xmin to
  xmax.  Discard overflows or underflows
  "
  [xmin xmax nbins xs]
  (let [;; "base" histogram (zeros):
        zero-map (into (sorted-map)
                       (map (fn [x] [x 0]) (range nbins)))
        ;; get actual bin values for every input in xs:
        xbins (map #(int (* nbins (/ (- % xmin)
                                     (- xmax xmin))))
                   xs)
        ;; strip out undeflows & overflows:
        no-overflows (->> xbins
                          (remove #(< % 0))
                          (remove #(>= % nbins)))]
    ;; yield histogram as array of [ibin, height] pairs:
    (into [] (reduce #(update-in %1 [%2] inc) zero-map no-overflows))))

(defn draw-hist
  "
  Draw histogram of bins as generated by make-hist
  "
  [x-label log? values]
  (let [renderer (XYBarRenderer.)
        painter (StandardXYBarPainter.)
        series (XYSeries. [])
        blue (java.awt.Color. 0x3b 0x6c 0x9d)
        coll (XYSeriesCollection. series)
        y-axis (if log?
                 (org.jfree.chart.axis.LogarithmicAxis. "Entries")
                 (org.jfree.chart.axis.NumberAxis. "Entries"))
        plot (XYPlot. coll (NumberAxis. x-label) y-axis renderer)
        panel (JFreeChart. plot)
        frame (ChartFrame. "Histogram" panel)]
    (doto plot
      (.setBackgroundAlpha 0.0)
      (.setRangeGridlinesVisible false)
      (.setDomainGridlinesVisible false))
    (doto renderer
      (.setBarPainter painter)
      (.setSeriesPaint 0 blue)
      (.setDrawBarOutline true)
      (.setSeriesOutlinePaint 0 blue)
      (.setShadowVisible false))
    (doseq [[x y] values]
      (when-not (and log? (zero? y))
        (.add series (+ x 0.5) y)))
    (.setLowerBound y-axis 0.5)
    (.setVisible (.getLegend panel) false)
    (doto frame
      (.setSize 800 300)
      (.setVisible true))))

(comment
  (->> [100 200 200 200 200 200 200 200 200 200 200 300]
       (make-hist 10 1000 100)
       (draw-hist "Simple Log Test" true))

  (->> [100 200 200 300]
       (make-hist 10 1000 100)
       (draw-hist "Simple Linear Test" false))

  (->> #(apply + (repeatedly 100 rand))
       (repeatedly 10000)
       (concat (repeatedly 1000 #(rand-int 400)))
       (make-hist 10 100 100)
       (draw-hist "Gaussian-ish, linear" false))

  (->> #(apply + (repeatedly 100 rand))
       (repeatedly 10000)
       (concat (repeatedly 1000 #(rand-int 400)))
       (make-hist 10 100 100)
       (draw-hist "Gaussian-ish, log" true)))
