(ns retabled.internal
  (:require [retabled.filter :as filter]
            [retabled.helpers :as helpers]
	    [retabled.sort :as sort]
            #?(:cljs [reagent.core :refer [atom]])
            [clojure.string :as str]))

(def exceptions {:duplicate-id "multiple tables with the same table-id"})

(def default-paging-info {:entries-per-page 0
                        :go-to-value "1"
                        :current-page 1})

(def PAGING-INFO (atom {}))

(def ALL-DATA (atom {}))

(def CURRENT-DATA (atom {}))

(defn atom?
  "ducktype an atom as something dereferable"
  [a]
  (try (do (deref a) true)
       (catch #?(:clj Exception :cljs js/Error) _ false)))

(defn render-header-fields
  [{:as args
    :keys [controls
           SORT
           FILTER
           table-id]}]
  (let [table-color (get-in controls [:table-scroll-bar :table-color])
        style-first-column-fixed {:style {"position" "sticky"
                                                        "left" "0"
                                          "backgroundColor" table-color}}
        style-last-column-fixed {:style {"position" "sticky"
                                                       "right" "0"
                                         "backgroundColor" table-color}}
        first-column-and-first-is-fixed? (fn [column] (and (= column (first (:columns controls))) (get-in controls [:table-scroll-bar :first?])))
        last-column-and-last-is-fixed? (fn [column] (and (= column (last (:columns controls))) (get-in controls [:table-scroll-bar :last?])))
        sort-filter-highlight (:sort-filter-highlight controls)
        currently-sorted-or-filtered? (fn [column] (or (and (:selected @SORT)(= (:sortfn column) (:selected @SORT)))
                                                       (= (:valfn column) (:selected @SORT))
                                                       (> (count (get-in @FILTER [(:valfn column) :value])) 0)))]
    (into [:tr.table-headers.row]
          (for [c (:columns controls)
                :let [h  (cond->> (:headline c)
                           (:sort c) (sort/gen-sort c SORT))
                      fi (when (:filter c) (filter/gen-filter {:col-map c
                                                               :FILTER  FILTER
                                                               :table-id  table-id
                                                               :filter-in-url (:filter-in-url controls)}))
                      style-th (assoc-in (if (first-column-and-first-is-fixed? c)
                                           style-first-column-fixed
                                           (if (last-column-and-last-is-fixed? c)
                                             style-last-column-fixed))
                                         [:style "backgroundColor"]
                                         (if (and sort-filter-highlight (currently-sorted-or-filtered? c))
                                           (if (string? sort-filter-highlight)
                                             sort-filter-highlight
                                             "rgb(240, 240, 240)")
                                           table-color))]]
            ^{:key (:genkey c)}
            [:th style-th fi h]))))

(defn render-screen-controls
  "Render the controls to edit this screen for results"
  [{:as paging-controls
    :keys [get-current-screen
           get-amount
           set-amount
           set-current-screen
           set-final-screen
           get-final-screen
           r-content
           rr-content
           f-content
           ff-content
           left-bar-content
           right-bar-content
           num-columns
           entries-option
           align
           show-total-pages?]
    :or {num-columns 100}}
   table-scroll-bar
   table-id]
  (let [current-screen-for-display (inc (get-current-screen))
        prevfn #(max (dec (get-current-screen)) 0)
        nextfn #(min (inc (get-current-screen)) (get-final-screen))
        first-page? (= (get-current-screen) 0)
        last-page? (= (get-current-screen) (get-final-screen))
        style-hidden {:style {"color" "transparent"}
                      :class "hidden"}
        style-page-to-go {:style {"width" "3em"
                                  "marginLeft" ".5em"}
                          :value (or (when (empty? (:go-to-value (get @PAGING-INFO table-id)))
                                       "")
                                     current-screen-for-display)
                          :on-click #(swap! PAGING-INFO assoc-in [table-id :go-to-value] "")
                          :id "page-to-go"}
        on-change-page-to-go (fn [evt]
                               (let [val (-> evt .-target .-value)
                                     int-val (int val)]
                                 (when (= val "")
                                   (swap! PAGING-INFO assoc-in [table-id :go-to-value] val))
                                 (when (and (> int-val 0)(<= int-val (+ (get-final-screen) 1)))
                                   (do (swap! PAGING-INFO assoc-in [table-id :go-to-value] (str int-val))
                                       (set-current-screen (- int-val 1))))))
        table-color (:table-color table-scroll-bar)
        sc-map {:colSpan num-columns :style {"borderColor" table-color "textAlign" align}}
        div-conditional-styling (when table-scroll-bar {:style {"position" "sticky" align "1em"}})
        screen-num-content (if show-total-pages?
                             (str current-screen-for-display " of " (+ (get-final-screen) 1))
                             current-screen-for-display)
        page-to-go-map (assoc style-page-to-go :on-change on-change-page-to-go)
        entries-option-and-values-set (and entries-option (:values entries-option))
        entries-option-change (fn [evt]
                                                       (let [val (int (-> evt .-target .-value))]
                                                         (swap! PAGING-INFO assoc-in [table-id :entries-per-page] val)
                                                         (set-current-screen 0)
                                                         (swap! PAGING-INFO assoc-in [table-id :go-to-value] "1")))]
    [:tr.row.screen-controls-row
     [:td.cell.screen-controls sc-map
      [:div.control div-conditional-styling
       left-bar-content
       [:div.control.first [:a.control-label (if first-page?
                                              style-hidden
                                              {:on-click #(set-current-screen 0)})
                           rr-content]]
      [:div.control.prev [:a.control-label (if first-page?
                                             style-hidden
                                             {:on-click #(set-current-screen (prevfn))})
                          r-content]]
       [:div.control.current-screen [:span.screen-num screen-num-content]]
      [:div.control.next [:a.control-label (if last-page?
                                             style-hidden
                                             {:on-click #(set-current-screen (nextfn))})
                          f-content]]
      [:div.control.final [:a.control-label (if last-page?
                                              style-hidden
                                              {:on-click #(set-current-screen (get-final-screen))})
                           ff-content]]
      [:span.go-to "Go to"]
       [:input.page-to-go page-to-go-map]
       (when entries-option-and-values-set
        (let [values (distinct (:values entries-option))
              default (if (:default entries-option)
                        (:default entries-option)
                        (first values))]
          (into [:select.entries-per-page {:defaultValue default
                                           :onChange entries-option-change
                                           :style {"marginLeft" "1em"}}]
                (for [num values
                      :let [val-map {:value num}]]
                  ^{:key num}
                  [:option val-map  num]))))]
      right-bar-content]]))

(defn generate-theads
  "generate the table headers"
  [{:as args
    :keys [controls
           paging-controls
           SORT FILTER
           table-id]}]
  (let [table-color (get-in controls [:table-scroll-bar :table-color])
        tsb-style {:style {"position" "sticky"
                       "top" "0"
                       "backgroundColor" table-color
                       "zIndex" "1"}}]
    [:thead (when (:table-scroll-bar controls)
              tsb-style)
     (when (:example/csv-download? controls)
     (helpers/render-csv-button-row @CURRENT-DATA table-id))
     (when (:paging controls)
       (render-screen-controls paging-controls (:table-scroll-bar controls) table-id))
     (render-header-fields {:controls controls
                            :SORT  SORT
                            :FILTER  FILTER
                            :table-id  table-id})]))

(defn generate-rows
  "Generate all the rows of the table from `entries`, according to `controls`"
  [{:as args
    :keys [controls
           entries SORT
           FILTER
           table-id]}]
  (let [{:keys [row-class-fn columns sort-filter-highlight]
         :or {row-class-fn (constantly "row")}} controls
        table-color (get-in controls [:table-scroll-bar :table-color])
        style-first-column-fixed {"position" "sticky"
                                  "left" "0"
                                  "backgroundColor" table-color}
        style-last-column-fixed {"position" "sticky"
                                 "right" "0"
                                 "backgroundColor" table-color}
        first-column-and-first-is-fixed? (fn [column] (and (= column (first (:columns controls))) (get-in controls [:table-scroll-bar :first?])))
        last-column-and-last-is-fixed? (fn [column] (and (= column (last (:columns controls))) (get-in controls [:table-scroll-bar :last?])))]
    (into [:tbody]
          (for [e entries :let [tr ^{:key (:genkey e)} [:tr {:class (row-class-fn e)}]]]
            (into tr
                    (for [c columns :let [{:keys [valfn css-class-fn displayfn filter]                                         
                                           :or {css-class-fn (constantly "field")
                                                displayfn identity}} c
                                          arg-map (cond-> {:class (css-class-fn e)}
                                                    (= filter :click-to-filter) (assoc :on-click (filter/on-click-filter {:col-map c
                                                                                                                          :table-id  table-id
                                                                                                                          :filter-in-url (:filter-in-url controls)
                                                                                                                          :FILTER  FILTER :value  (filter/resolve-filter c e)}))
                                                    (= filter :click-to-filter) (assoc :class (str (css-class-fn e) " click-to-filter")))
                                          currently-sorted-or-filtered? (or (and (:selected @SORT)(= (:sortfn c) (:selected @SORT)))
                                                                            (= valfn (:selected @SORT))
                                                                            (> (count (get-in @FILTER [valfn :value])) 0))
                                          cell-style-and-arg-map (assoc-in (if (first-column-and-first-is-fixed? c)
                                                                             (assoc arg-map :style style-first-column-fixed)
                                                                             (if (last-column-and-last-is-fixed? c)
                                                                               (assoc arg-map :style style-last-column-fixed)
                                                                               arg-map))
                                                                           [:style "backgroundColor"]
                                                                           (if (and sort-filter-highlight currently-sorted-or-filtered?)
                                                                             (if (string? sort-filter-highlight)
                                                                               sort-filter-highlight
                                                                               "rgb(240, 240, 240)")
                                                                             table-color))]]
                      ^{:key (:genkey c)} [:td.cell cell-style-and-arg-map
                                           (-> e valfn displayfn)]))))))

(def default-page-atom {:current-screen 0
                              :final-screen 0
                              :per-screen 10})

(def DEFAULT-PAGE-ATOM (atom {}))

(defn default-paging
  "Set up a local atom and define paging functions with reference to it"
  [table-id]
  (let [paging {:get-current-screen #(if (< (:current-screen (get @DEFAULT-PAGE-ATOM table-id)) (:final-screen (get @DEFAULT-PAGE-ATOM table-id)))
                                       (:current-screen (get @DEFAULT-PAGE-ATOM table-id))
                                       (do (swap! PAGING-INFO assoc-in [table-id :current-page] (+ 1 (:final-screen (get @DEFAULT-PAGE-ATOM table-id))))
                                           (when (not-empty (:go-to-value (get @PAGING-INFO table-id)))
                                             (swap! PAGING-INFO assoc-in [table-id :go-to-value] (str (+ 1 (:final-screen (get @DEFAULT-PAGE-ATOM table-id))))))
                                           (:final-screen (get @DEFAULT-PAGE-ATOM table-id))))
                :set-current-screen #(do (swap! DEFAULT-PAGE-ATOM assoc-in [table-id :current-screen] %)
                                         (swap! PAGING-INFO assoc-in [table-id :current-page] (+ 1 %))
                                         (swap! PAGING-INFO assoc-in [table-id :go-to-value] (str (+ 1 %))))
                :get-amount #(if (> (:entries-per-page (get @PAGING-INFO table-id)) 0)
                               (:entries-per-page (get @PAGING-INFO table-id))
                               (:per-screen (get @DEFAULT-PAGE-ATOM table-id)))
                :set-amount #(swap! DEFAULT-PAGE-ATOM assoc-in [table-id :per-screen] %)
                :get-final-screen #(:final-screen (get @DEFAULT-PAGE-ATOM table-id))
                :set-final-screen #(swap! DEFAULT-PAGE-ATOM assoc-in [table-id :final-screen] %)
                :r-content "‹"
                :rr-content "«"
                :f-content "›"
                :ff-content "»"
                :align "left"}]
    paging))

(defn paging
  "Limit view of entries to a given screen.
  If `paging-controls` is falsy, do not filter."
  [paging-controls entries]
  (if-not paging-controls entries
          (let [{:keys [get-current-screen
                        get-amount
                        set-amount
                        set-current-screen
                        set-final-screen
                        get-final-screen]} paging-controls
                amt (get-amount)
                parted-entries (if (> amt (count entries))
                                 (list entries)
                                 (partition amt amt nil entries))
                max-screens (dec (count parted-entries))]                  
              (set-final-screen max-screens)
              (nth parted-entries (get-current-screen)))))

(defn curate-entries
  [{:as args
    :keys [paging-controls entries SORT FILTER]}]
  (when (not-empty entries)
    (->> entries
         (filter/filtering FILTER)
         (sort/sorting SORT)
         (paging paging-controls))))

(def TABLE-NAME-COUNT (clojure.core/atom 0))

(defn check-for-duplicates
  [table-id]
  (when (contains? @ALL-DATA table-id)
    (throw #?(:clj (Exception. (:duplicate-id exceptions)) :cljs (js/Error. (:duplicate-id exceptions))))))

(defn generate-table-id
  [table-id]
  (if-not table-id
    (do
      (swap! TABLE-NAME-COUNT inc)
      (str "__retabled-" @TABLE-NAME-COUNT))
    (do
      (try
        (check-for-duplicates table-id)
        (catch #?(:clj Exception :cljs js/Error) err
          #?(:clj (println err) :cljs (js/console.error err))))
      table-id)))

(defn update-all-data!
  [table-id entries]
  (when table-id
    (swap! ALL-DATA assoc table-id entries)))

(defn update-current-data!
  [table-id entries]
  (when table-id
    (swap! CURRENT-DATA assoc table-id entries)))

(defn update-default-entries-per-page!
  [controls table-id]
  (let [entries-option (get-in controls [:paging :entries-option])
        default (get entries-option :default)
        values (get entries-option :values)
        num (if default
              default
              (first values))]
    (when (and entries-option num)
      (swap! PAGING-INFO assoc-in [table-id :entries-per-page] num))))

(defn update-paging-info!
  [table-id]
  (swap! PAGING-INFO assoc table-id default-paging-info)
  (swap! DEFAULT-PAGE-ATOM assoc table-id default-page-atom))
