(ns idm.graph.op "For keywords")

(ns idm.graph.op.type "Also for keywords")

(ns idm.graph.attributes
  "Functions for interacting with graphs that represent attributes"
  (:require
    [clojure.tools.logging :as log]
    [idm.graph.op :as op]
    [idm.graph.op.type :as op.type]
    [idm.graph.relation :as rel]
    [idm.graph.relation.type :as rel.type]
    [ubergraph.core :as uber]))

(defn require-attr
  "Generate a predicate function that, when applied to a node or edge, will
  indicate whether or not the node or edge has the specified value for the
  given attribute."
  [graph attr value]
  (fn [x] (= value (uber/attr graph x attr))))

(defn resolving-edge-pred
  "Generate a predicate function that indicates whether or not a given edge in
  a graph is marked as a resolver."
  [graph]
  (require-attr graph ::op/type ::op.type/resolve))

(defn ident-resolver
  "Creates a function that will extract the given attribute from the context,
  if possible, and yield an ident of the form `[attr value]`."
  [attr]
  (fn [ctx]
    (when-some [v (ctx attr)]
      [attr v])))

(defn ident-wrapper
  "Like ident-resolver, but assumes the value is being pulled out of the
  context by the caller."
  [attr]
  (fn [value] [attr value]))

(defn ident-edges
  "Generate a set of edges from `src-attrs` to `ident-attr`, with appropriate
  relations specified."
  [src-attrs ident-attr]
  (for [attr src-attrs]
    [attr ident-attr
     {::op/type   ::op.type/derive
      ::op/fn     (ident-wrapper attr)
      ::rel/type  ::rel.type/one->one
      ::rel/id    (keyword (namespace attr) (str (name attr) "->ident"))}]))

(defn resolver-edges
  "Generate the set of relations and edges required to define a given resolver.

  If not provided, generates an ident node based on one of the given
  `src-attrs` as `<attr namespace>/ident`. For each node in `src-attrs`,
  generates an edge to the ident node of type `derive`. For each node in
  `dest-attrs`, generates an edge from the ident node to that node; these will
  all be a part of the same `resolve` relation.

  If only `src-attrs` and `dest-attrs` are provided, generates an ident key
  in the namespace of the the first keyword in `src-attrs`.

  If `resolver-id` is not provided, produces it by adding `#0` to the name of
  the ident key. Provide this manually if it is not globally unique."
  ([src-attrs dest-attrs]
   (->> (keyword (namespace (first src-attrs)) "ident")
        (log/spyf :debug "Generated ident keyword: %s")
        (resolver-edges src-attrs dest-attrs)))
  ([src-attrs dest-attrs ident-kw]
   (->> (keyword (namespace ident-kw) (str (name ident-kw) "#0"))
        (log/spyf :debug "Generated relation id %s based on ident key")
        (resolver-edges src-attrs dest-attrs ident-kw)))
  ([src-attrs dest-attrs ident-kw resolver-id]
   (concat
     (for [attr src-attrs]
       [attr ident-kw {::op/type  ::op.type/derive
                       ::rel/type ::rel.type/one->one
                       ::rel/id   (keyword (namespace attr)
                                           (str (name attr)
                                                "->ident"))
                       ::op/fn    (ident-wrapper attr)
                       ;; indicate this edge should use a generic ident fn
                       ::op/->ident? true}])
     ;; resolve fns for one->many nodes should be on the node
     (for [attr dest-attrs]
       [ident-kw attr {::op/type  ::op.type/resolve
                       ::rel/type ::rel.type/one->many
                       ::rel/id   resolver-id}]))))
