(ns idm.graph.util
  "Utility functions for graph management"
  (:require
    [clojure.spec.alpha :as s]
    [clojure.tools.logging :as log]
    [idm.graph.keyword :as kw]
    [idm.graph.multipath :as mp]
    [idm.graph.spec]
    [ubergraph.core :as uber]))

(defn valid-multipath?
  "Validate that the sequence of steps indicated by a multipath is actually
  traversable"
  ;; TODO: basically throwing away information
  [start-nodes multipath]
  ; get edge sets without unwrapping
  (let [edge-sets (mp/edge-sets multipath)
        res
        (reduce
          (fn [seen-nodes next-edges]
            (if-let [missing (first (remove #(contains? seen-nodes %)
                                            (map uber/src next-edges)))]
              (reduced missing)
              (into seen-nodes (map uber/dest) next-edges)))
          (set start-nodes)
          edge-sets)]
    (set? res)))

(defn e->str
  "Debug representation of an edge. Ignores attributes."
  [e]
  (if (uber/edge? e)
    (->> ((juxt uber/src uber/dest) e)
         (mapv #(cond-> (str %)
                  (keyword? %) (subs 1)))
         (apply format "%s->%s"))
    (str e)))
(s/fdef e->str
  :args :idm.graph/edge
  :ret  string?)

(defn edge->vector
  "Convert an Ubergraph edge to a vector of the form [src dest attrs], using
  the graph, which is stored on a given edge as metadata (usually) to retrieve
  the attributes."
  [edge]
  [(uber/src edge)
   (uber/dest edge)
   (when-let [graph (not-empty (meta edge))]
     (uber/attrs graph edge))])

(s/fdef edge->vector
  :args :idm.graph/edge
  :ret  (s/tuple :idm.graph/node :idm.graph/node (s/nilable map?)))

(defn- compile-alias
  "Compiles a vector of alias edges from two args, where the first is the
  source and the second is either the destination or a collection of
  destinations.

  Examples:
  ```clojure
  (compile-alias :a :b)
  => [[:a :b]]
  (compile-alias :a [:b :c :d])
  => [[:a :b]
  [:a :c]
  [:a :d]]
  ```"
  [src dst]
  {:pre [(keyword? src)
         (or (keyword? dst)
             (every? keyword? dst))]}
  (if (coll? dst)
    (mapv #(vector src % {::alias? true
                          ::op :idm.graph.op/derive
                          ::fn src})
          dst)
    (compile-alias src [dst])))

(defn add-resolvers
  "Adds a set of nodes and edges as follows:

  1. A generated ident node matching the target namespace. This node will be
  marked as a resolver, and have the resolve function stored in its attrs.
  2. Edges from `unique-attrs` to the ident node
  3. Edges from the generated ident node to `all-attrs`"
  [g unique-attrs all-attrs resolve-fn]
  (let [ident-kw (kw/qualify (first unique-attrs) :ident)]
    ;(keyword
    ;  (str (namespace (first unique-attrs))
    ;       ".virtual")
    ;  "ident")]
    (log/debug "Generated ident key " ident-kw)
    (-> g
        ; ident node
        (uber/add-nodes-with-attrs
          [ident-kw {::resolver? true
                     ::fn resolve-fn}])
        ;(resolve-ident ident-kw)}])
        ; edges from unique to ident
        (uber/add-edges*
          (mapv (fn [k]
                  [k ident-kw {::op :idm.graph.op/derive
                               ::fn #(vector k (get % k))}])
                unique-attrs))
        ; edges from ident to all
        (uber/add-edges*
          (mapv (fn [k] [ident-kw k {::op :idm.graph.op/resolve}]) all-attrs)))))
