(ns idm.graph.traverse
  "Functions to allow for actually traversing a planned path."
  (:require
    [clojure.tools.logging :as log]
    [idm.graph.multipath :as mp]
    [idm.graph.op :as op]
    [idm.graph.op.type :as op.type]
    [idm.graph.relation :as rel]
    [idm.graph.relation.type :as rel.type]
    [ubergraph.core :as uber]))

(defn execute
  "Given a multipath, walks the relations and executes operations indicated by
  their attributes, providing context to the operations and storing the results
  in the growing context map."
  ([multipath]
   (log/debug "No context provided; using empty")
   (execute multipath {}))
  ([multipath ctx]
   (reduce
     (fn [ctx rel]
       (log/tracef "Context meta map: %s" (meta ctx))
       (let [{::rel/keys [id type traversed-edges srcs dests]} rel
             dests (or (not-empty 
                         (into #{} (map uber/dest) traversed-edges))
                       dests)
             ;; TODO: is it even worth checking for input existence, or
             ;; does it make more sense to check for outputs, thus ensuring
             ;; input? (more or less)
             resolve-fn (if (= ::op.type/alias (::op/type rel))
                          identity
                          (::op/fn rel))
             ;; used to mark (roughly) where values come from
             tag (select-keys rel [::rel/id ::rel/type ::op/type])
             ;; Use ONLY when one source is known
             src (first srcs)
             ;; similarly, use ONLY when one destination is known
             dest (first dests)]
         (log/tracef "Evaluating relation %s with initial context %s" id ctx)
         (log/debugf "Resolving a %s relation from %s to %s" 
                     (name type) srcs dests)
         (case type
           ;; one->one resolvers should take a one argument, the input value
           ::rel.type/one->one
           (let [res (resolve-fn (get ctx src))]
             (log/debugf "Resolved %s=%s based on %s" dest res src)
             (-> (assoc ctx dest res)
                 (vary-meta assoc dest tag)))
           ;; one->many should take the input value and the desired outputs
           ::rel.type/one->many
           (let [res (resolve-fn (get ctx src) dests)]
             (log/debugf "Resolved outputs from %s: %s" src res)
             ;; TODO: best way to do it?
             (assert (map? res) "one->many output must be a map")
             (-> ctx ;; rather than zipmap dests with tag, only tag retrieved
                 (vary-meta into (map #(vector % tag)) (keys res))
                 (into res)))
           ;; many->one should take one input, a map of values
           ::rel.type/many->one
           (let [res (resolve-fn (select-keys ctx srcs))]
             (log/debugf "Resolved %s=%s based on %s" dest res srcs)
             (-> (assoc ctx dest res)
                 (vary-meta assoc dest tag))))))
     ctx
     (mp/relations multipath))))
