(ns idm.graph.relation.type
  "Declaring for use of shorthand keywords")

(ns idm.graph.relation
  "Functions and spec declarations for graph relations"
  (:require [clojure.spec.alpha :as s]
            [idm.graph.relation.type :as rel.type]
            [idm.graph.util :as util]
            [ubergraph.core :as uber]))

(s/def ::id keyword?)

(s/def ::type #{::rel.type/one->one
                ::rel.type/one->many
                ::rel.type/many->one})

;; TODO: only one expected for one->one
(s/def ::edges 
  (s/coll-of :idm.graph/edge :distinct true :into #{} :min-count 1))

(s/def ::visited-edges 
  (s/coll-of :idm.graph/edge :distinct true :into #{}))

(s/def ::srcs (s/coll-of :idm.graph/node :kind set?))

(s/def ::dests (s/coll-of :idm.graph/node :kind set?))

(s/def :idm.graph/relation
  (s/keys :req [::id ::type ::edges ::srcs ::dests] :opt [::visited-edges]))

(defn edge->id
  "Generate a relation id for and edge based on its src, dest, and internal
  ubergraph id. Do not use for edges that belong to a relation with an ID;
  this will not produce that."
  [edge]
  (keyword (str (util/e->str edge) \# (:id edge))))
(s/fdef edge->id
  :args :idm.graph/edge
  :ret   ::id)

(defn- compile-srcs-dests
  "Add the sources and destinations of its edges to a given relation."
  [rel]
  (let [{::keys [edges]} rel]
    (assoc rel
           ::srcs   (into #{} (map uber/src) edges)
           ::dests  (into #{} (map uber/dest) edges))))

(defn to-edges
  "Given a relation, return a sequence of edges suitable for adding to a graph
  to implement the relation."
  [rel]
  (let [rel (update rel ::type #(or % ::rel.type/one->one))
        {::keys [srcs dests type id]} rel]
    (assert (or id (= type ::rel.type/one->one))
            "relation must either be one->one or declare a relation ID")
    (assert (case type
              ::rel.type/one->one (= 1 (count srcs) (count dests))
              ::rel.type/one->many (= 1 (count srcs))
              ::rel.type/many->one (= 1 (count dests)))
            "relation should have appropriate src/dest counts for type")
    (for [src srcs
          dest dests]
      [src dest rel])))

(defn from-edge
  "Get the relation an edge belongs to. Relation will include type, id, and
  member edges.

  When `graph` is not provided, assumes you know this is a one->one edge and
  completes as such."
  ([edge]
   {::type  ::rel.type/one->one
    ::id    (edge->id edge)
    ::edges #{edge}})
  ([g edge]
   (let [{::keys [id type] :as rel} (update (uber/attrs g edge)
                                            ::type #(or % ::rel.type/one->one))
         edges (->> (case type
                      ::rel.type/one->one #{edge}
                      ::rel.type/one->many (uber/out-edges g (uber/src edge))
                      ::rel.type/many->one (uber/in-edges g (uber/dest edge)))
                    (into #{} (filter #(= id (uber/attr g % ::id)))))]
     (assert (or id (= type ::rel.type/one->one))
             (str "Edges must be either one->one or declare a relation ID"
                  \space rel))
     ;; can now update id if necessary
     (-> rel
         (update ::id #(or % (edge->id edge)))
         (assoc ::edges edges)
         compile-srcs-dests))))

(s/fdef from-edge
  :args (s/alt :only-edge   (s/cat :edge :idm.graph/edge)
               :graph-edge  (s/cat :graph :idm.graph/graph
                                   :edge  :idm.graph/edge))
  :ret  :idm.graph/relation)
