(ns idm.graph.relation.type
  "Declaring for use of shorthand keywords")

(ns idm.graph.relation
  "Functions and spec declarations for graph relations"
  (:require [clojure.spec.alpha :as s]
            [idm.graph.relation.type :as rel.type]
            [idm.graph.util :as util]
            [ubergraph.core :as uber]))

(s/def :idm.graph/edge uber/edge?)

(s/def ::id keyword?)

(s/def ::type #{::rel.type/one->one
                ::rel.type/one->many
                ::rel.type/many->one})

;; TODO: only one expected for one->one
(s/def ::edges 
  (s/coll-of :idm.graph/edge :distinct true :into #{} :min-count 1))

(s/def ::visited-edges 
  (s/coll-of :idm.graph/edge :distinct true :into #{}))

(s/def :idm.graph/relation
  (s/keys :req [::id ::type ::edges] :opt [::visited-edges]))

(defn edge->id
  "Generate a relation id for and edge based on its src, dest, and internal
  ubergraph id. Do not use for edges that belong to a relation with an ID;
  this will not produce that."
  [edge]
  (keyword (str (util/e->str edge) \# (:id edge))))
(s/fdef edge->id
  :args :idm.graph/edge
  :ret   ::id)

(defn from-edge
  "Get the relation an edge belongs to. Relation will include type, id, and
  member edges.

  When `graph` is not provided, assumes you know this is a one->one edge and
  completes as such."
  ([edge]
   {::type  ::rel.type/one->one
    ::id    (edge->id edge)
    ::edges #{edge}})
  ([graph edge]
   (let [{::keys [type id] :as rel} (-> (uber/attrs graph edge)
                                        (select-keys [::type ::id])
                                        (update ::type 
                                                #(or % ::rel.type/one->one)))]
     (assert (or id (= type ::rel.type/one->one))
             (str "Edges must be either one->one or declare a relation ID"
                  \space rel))
     (if (nil? id)
       ;; one->one edge without real info; delegate to 1-arity
       (from-edge edge)
       ;; otherwise just add sibling edges
       (->> (case type
              ::rel.type/one->one #{edge}
              ::rel.type/one->many (uber/out-edges graph (uber/src edge))
              ::rel.type/many->one (uber/in-edges graph (uber/dest edge)))
            (into #{} (filter #(= id (uber/attr graph % ::id))))
            (assoc rel ::edges))))))
