(ns idm.graph.util
  "Utility functions for graph management"
  (:require
    [clojure.tools.logging :as log]
    [idm.graph.alg :as alg]
    [idm.graph.keyword :as kw]
    [ubergraph.core :as uber]))

(defn valid-multipath?
  "Validate that the sequence of steps indicated by a multipath is actually
  traversable"
  ;; TODO: basically throwing away information
  [start-nodes multipath]
  ; get edge sets without unwrapping
  (let [edge-sets (alg/edge-sets multipath)
        res
        (reduce
          (fn [seen-nodes next-edges]
            (if-let [missing (first (remove #(contains? seen-nodes %)
                                            (map uber/src next-edges)))]
              (reduced missing)
              (into seen-nodes (map uber/dest) next-edges)))
          (set start-nodes)
          edge-sets)]
    (set? res)))

(defn- compile-alias
  "Compiles a vector of alias edges from two args, where the first is the
  source and the second is either the destination or a collection of
  destinations.

  Examples:
  ```clojure
  (compile-alias :a :b)
  => [[:a :b]]
  (compile-alias :a [:b :c :d])
  => [[:a :b]
  [:a :c]
  [:a :d]]
  ```"
  [src dst]
  {:pre [(keyword? src)
         (or (keyword? dst)
             (every? keyword? dst))]}
  (if (coll? dst)
    (mapv #(vector src % {::alias? true
                          ::op :idm.graph.op/derive
                          ::fn src})
          dst)
    (compile-alias src [dst])))

(defn add-resolvers
  "Adds a set of nodes and edges as follows:

  1. A generated ident node matching the target namespace. This node will be
  marked as a resolver, and have the resolve function stored in its attrs.
  2. Edges from `unique-attrs` to the ident node
  3. Edges from the generated ident node to `all-attrs`"
  [g unique-attrs all-attrs resolve-fn]
  (let [ident-kw (kw/qualify (first unique-attrs) :ident)]
    ;(keyword
    ;  (str (namespace (first unique-attrs))
    ;       ".virtual")
    ;  "ident")]
    (log/debug "Generated ident key " ident-kw)
    (-> g
        ; ident node
        (uber/add-nodes-with-attrs
          [ident-kw {::resolver? true
                     ::fn resolve-fn}])
        ;(resolve-ident ident-kw)}])
        ; edges from unique to ident
        (uber/add-edges*
          (mapv (fn [k]
                  [k ident-kw {::op :idm.graph.op/derive
                               ::fn #(vector k (get % k))}])
                unique-attrs))
        ; edges from ident to all
        (uber/add-edges*
          (mapv (fn [k] [ident-kw k {::op :idm.graph.op/resolve}]) all-attrs)))))
