(ns idm.graph.keyword
  "Utility functions for handling keywords. 
  
  Functions are memoized in situations where they handle at most one or two
  keywords directly (at a time, that is), as opposed to operating on
  collections."
  (:refer-clojure :exclude [name namespace])
  (:require
    [clojure.core :as core]
    [clojure.string :as str]))

(defn- flatten-map
  "Takes a nested map and returns a single-level map indexed by vectors.

  `path` *must* be a vector."
  ([m] (flatten-map [] m))
  ([path m]
   (let [flatten-1 (fn [[k v]]
                     (let [path (conj path k)]
                       (if (and (map? v) (not-empty v))
                         (flatten-map path v)
                         [path v])))]
     (into {} (map flatten-1) m))))

(defn name
  "Like [[core/name]], but returns a keyword, and is `nil`-safe.

  Example:
  ```clojure
  (mapv kw/name [:a/b :c nil])
  => [:b :c nil]
  ```"
  [x]
  (some-> x core/name keyword))

(defn namespace
  "Like [[core/namespace]], but returns a keyword, and is `nil`-safe.

  Example:
  ```clojure
  (mapv kw/namespace [:a/b :c nil])
  => [:a nil nil]
  ```"
  [x]
  (some-> x core/namespace keyword))

(defn as-tuple
  "Split a keyword into a tuple of [ns body], both keywords.
  Examples:
  ```clojure
  (mapv ns/as-tuple [:some-ns/some-kw :some.kw nil])
  => [[:some-ns :some-kw]
      [nil :some.kw]
      [nil nil]]
  ```"
  [kw]
  ((juxt namespace name) kw))

(defn split
  "Splits a keyword into a vector of keywords where the first element is the
  namespace or nil, and the rest is the body of the keyword split by `re`;
  defaults to splitting on '.'.

  :some/big.keyword => [:some :big :keyword]

  Does not split keyword namespace. Also works for strings delimited by '.', or
  indeed for symbols as well, probably."
  ([kw] (split kw #"\."))
  ([kw delim]
   (let [[ns body :as tup] (as-tuple kw)]
     (if (nil? body)
       tup
       (into [ns] (map keyword) (str/split (core/name body) delim))))))

(defn qualify
  ; TODO: would it be better to qualify keywords to sub-namespace when
  ; qualified kw provided?
  ; (kw/qualify :a.b/c :d.e/f) => :a.b.c/f
  "Qualify a keyword with a given namespace, overwriting an existing ns.

  If `ns` is a qualified keyword, uses its namespace.

  Returns a partial when no keyword is provided."
  ([ns] (partial qualify ns))
  ([ns kw]
   (keyword
     (when ns
       (if (qualified-ident? ns) (core/namespace ns) (core/name ns)))
     (core/name kw))))

(defn qualify-all
  "Prefix a vector of keywords with a given namespace."
  ([ns] (map (qualify ns)))
  ([ns v]
   (into [] (qualify-all ns) v)))

(defn qualify-keys
  "Qualify each key in a map with a given namespace.

  If only the namespace is provided, returns a transducer suitable for use in
  `(into {} xf m)`."
  ([ns] (map (fn [[k v]] [(qualify ns k) v])))
  ([ns m]
   (into {} (qualify-keys ns) m)))

(defn join
  "Join a vector of keywords into a composite keyword.
  
  The following forms are equivalent:
  ```clojure
  ; :a/b.c.d
  (join :a :b :c :d)
  (join [:a :b :c :d])
  (join :a [:b :c :d])
  ```"
  ([kws]
   (cond
     (keyword? kws) kws
     ; this causes single-keyword vectors to be caught by the above clause
     (coll? kws) (apply join kws)
     :else (throw (ex-info "Wrong argument type" {:got kws
                                                  :expected #{keyword? coll?}}))))
  ([kw-ns & kws]
   (some->> (if (coll? (first kws)) (first kws) kws)
            (mapv name)
            (str/join \.)
            (qualify kw-ns))))

(defn flatten-join
  "Flatten a nested map into a single-level map indexed by composite keywords.

  May provide a base path as a vector; see examples.

  Example:
  ```clojure
  (def some-map
    {:a {:m {:n 1
             :l 2}}
     :b {:x 3
         :y 4}})

  (flatten-join some-map)
  => {:a/m.n  1
      :a/m.l  2
      :b/x    3
      :b/y    4}

  (flatten-join [:some.ns] some-map)
  => {:some.ns/a.m.n  1
      :some.ns/a.m.l  2
      :some.ns/b.x    3
      :some.ns/b.y    4}
  ```"
  ([m] (flatten-join [] m))
  ([path m]
   (->> m
        (flatten-map path)
        (into {} (map (fn [[k v]] [(join k) v]))))))

(defn member?
  "Checks that a symbol or keyword is a member of the given namespace.

  Namespace can be given as either a string or a keyword."
  [ns k]
  (and ns k (= (name ns) (namespace k))))

(defn share-ns?
  "Checks that two symbols or keywords are part of the same namespace."
  [x y]
  (member? (namespace x) y))
