(ns com.edocu.help.handlers.protocols
  (:require [clojure.tools.logging :as log]
            [io.pedestal.interceptor.helpers :as interceptors]
            [io.pedestal.interceptor.chain :as chain]
            [clojure.core.async :as async :refer [put! close!]]
            [ring.util.response :as ring-response]
            [cheshire.core :as json]
            [environ.core :as e]
            [buddy.auth.middleware :as buddy]
            [buddy.auth.backends :as backends]))

(defn secret []
  (or (e/env :secret) "APfcwZVfuOuquZKIolzb7wGsDyUhm7lo"))

(defprotocol Routes
  (routes [this] "Return routes"))

(defn nginx-pedestal-handler
  "Return Pedestal Interceptor which send response to nginx downstream"
  []
  (interceptors/after
    ::nginx-pedestal-handler
    (fn [{:keys [response response-channel] :as context}]
      (log/trace "nginx-pedestal-handler.")
      (put! response-channel response (fn [_] (close! response-channel)))
      context)))

(defn user-uid
  "Return UID of user from request"
  [request]
  (or
    (ring-response/get-header request "uid")
    "open"))

(defn- slurp-body-fn
  [{:keys [request response-channel] :as context}]
  (try
    (assoc context
      :body
      (json/parse-stream
        (clojure.java.io/reader (:body request))
        true))
    (catch Exception e
      (log/error e "slurp-body")
      (let [response (-> (ring-response/response "")
                         (ring-response/status 400))]
        (put! response-channel response (fn [_] (close! response-channel)))
        (-> context
            (assoc :response response)
            (chain/terminate))))))

(defn slurp-body
  "Return Pedestal Interceptor which slurp body of request"
  []
  (interceptors/before
    ::slurp-body #'slurp-body-fn))

(defn- request-identity-fn
  [{:keys [request response-channel] :as context}]
  (try
    (assoc context
      :user
      (buddy/authenticate-request request [(backends/jws {:secret (secret)})]))
    (catch Exception e
      (log/error e "request-identity")
      (let [response (-> (ring-response/response "")
                         (ring-response/status 400))]
        (put! response-channel response (fn [_] (close! response-channel)))
        (-> context
            (assoc :response response)
            (chain/terminate))))))

(defn request-identity
  "Return Pedestal Interceptor which slurp identity from request"
  []
  (interceptors/before
    ::request-identity #'request-identity-fn))
