(ns com.edocu.help.db.mongodb
  (:require [com.stuartsierra.component :as component]
            [clojure.tools.logging :as log]
            [environ.core :as e]
            [clojure.core.async :as async :refer [chan close!]])
  (:import [com.mongodb ReadPreference WriteConcern ConnectionString]
           [com.mongodb.async.client MongoClients MongoClient MongoDatabase MongoCollection]
           [com.mongodb.client.model Indexes IndexOptions]
           [com.mongodb.async SingleResultCallback]
           [org.bson Document]
           [clojure.lang IPersistentMap Named Keyword Ratio]
           [java.util List ArrayList Date Set]
           [org.joda.time DateTime]))


(defprotocol ConvertFromDocument
  (document->hashmap [input] "Converts given Document instance to a piece of Clojure data"))

(extend-protocol ConvertFromDocument
  nil
  (document->hashmap [input] input)

  Object
  (document->hashmap [input] input)

  List
  (document->hashmap [^List input]
    (vec (map #(document->hashmap %) input)))

  ArrayList
  (document->hashmap [^ArrayList input]
    (vec (map #(document->hashmap %) input)))

  Document
  (document->hashmap [^Document input]
    (reduce (fn [m ^String k]
              (assoc m (keyword k) (document->hashmap (.get input k))))
            {}
            (.keySet input))))

(defprotocol ConvertToDocument
  (hashmap->document [input] "Converts given piece of Clojure data to Document MongoDB Java async driver uses"))

(extend-protocol ConvertToDocument
  nil
  (hashmap->document [_]
    nil)

  String
  (hashmap->document [^String input]
    input)

  Boolean
  (hashmap->document [^Boolean input]
    input)

  Date
  (hashmap->document [^Date input]
    input)

  DateTime
  (hashmap->document [^DateTime input]
    (hashmap->document (.toDate input)))

  Ratio
  (hashmap->document [^Ratio input]
    (double input))

  Keyword
  (hashmap->document [^Keyword input] (name input))

  Named
  (hashmap->document [^Named input] (name input))

  IPersistentMap
  (hashmap->document [^IPersistentMap input]
    (let [o (Document.)]
      (doseq [[k v] input]
        (.put o (hashmap->document k) (hashmap->document v)))
      o))

  List
  (hashmap->document [^List input] (map hashmap->document input))

  Set
  (hashmap->document [^Set input] (map hashmap->document input))

  Document
  (hashmap->document [^Document input] input)

  Object
  (hashmap->document [input]
    input))


(defn ^ConnectionString replicas-uri []
  (ConnectionString.
    (e/env :mongo-uri)))

(defprotocol Collections
  (^MongoCollection collection
    [this name]
    [this name read-preferences]
    "Return instance of mongodb collection"))

(defprotocol IClient
  (close-connection! [this] "Close MongoDB client"))

(defrecord AsyncMongo [db client]
  component/Lifecycle
  (start [this]
    (log/trace "Starting MongoDB client")
    (let [c (delay (MongoClients/create (replicas-uri)))]
      (assoc this
       :client c
       :db (delay (let [^MongoDatabase database (.getDatabase @c (e/env :mongo-db))]
                    database)))))
  (stop [this]
    (log/trace "Stopping MongoDB client")
    (dissoc this :db))

  IClient
  (close-connection! [_]
    (.close @client))

  Collections
  (collection [this name]
    (collection this name (ReadPreference/secondaryPreferred)))
  (collection [this name read-preference]
    (doto (.getCollection
            ^MongoDatabase @db
            name)
      (.withReadPreference read-preference))))

(defn ->asyc-mongo
  []
  (map->AsyncMongo {}))

(defn <init-ttl-index!
  "Init TTL index on registration collection"
  [^MongoCollection collection ^IndexOptions index-options ^String index-key]
  (let [result (chan)
        callback (reify SingleResultCallback
                   (onResult [_ _ t]
                     (try
                       (if t (throw t))
                       (catch Exception e
                         (log/error e "<init-ttl-index!"))
                       (finally
                         (close! result)))))]
    (.createIndex
      collection
      (Indexes/ascending ^List [index-key])
      index-options
      callback)
    result))
