(ns com.edocu.elements.type.core
  (:use [com.edocu.elements.type.protocols])
  (:require [com.edocu.elements.type.http :as http]
            [clojure.core.async :refer [go <! >! chan close! thread promise-chan]]
            [clojure.tools.logging :as log]
            [clojure.core.memoize :as core-memo]
            [com.edocu.help.sentry :as sentry]
            [com.edocu.help.cache :as cache])
  (:import [com.edocu.elements.type.protocols
            ElementTypes
            ElementType]))

(defn- structure-load [element_type]
  (let [result (promise-chan)
        response_chan (chan)]
    (go
      (try
        (http/element-structure
          element_type
          response_chan)
        (let [structure_response (<! response_chan)]
          (log/trace "structure:" (pr-str structure_response))
          (if (some? structure_response)
            (>! result
                structure_response))
          (close! result))
        (catch Exception e
          (sentry/put-in-mdc {:element_type element_type})
          (log/error e "structure. element_type:" element_type)
          (close! result))))
    result))

(defn- filter-attribute [attribute_type structure_chan]
  (let [result_c (promise-chan)]
    (go
      (let [{:keys [attributes]} (<! structure_chan)
            result (set (map
                          (fn [[att _]]
                            att)
                          (filter
                            (fn [[_ {:keys [type]}]]
                              (= attribute_type type))
                            attributes)))]
        (log/debug "filter-attribute:" "attribute_type:" attribute_type "attributes:" attributes "result:" result)
        (>! result_c
            result)
        (close! result_c)))
    result_c))

(def ^:private filter-file-attribute (partial filter-attribute "File"))
(def ^:private filter-text-attribute (partial filter-attribute "Text"))
(def ^:private filter-dropdown-attribute (partial filter-attribute "Dropdown"))
(def ^:private filter-text-area-attribute (partial filter-attribute "Textarea"))
(def ^:private filter-date-time-attribute (partial filter-attribute "DateTime"))
(def ^:private filter-date-attribute (partial filter-attribute "Date"))

(defn- type-manager-impl-factory []
  {:structure            (fn [element_type]
                           @(:structure_delay element_type))

   :file-attributes      (core-memo/memo
                           (fn [element_type]
                             (filter-file-attribute
                               (structure element_type))))

   :text-attributes      (core-memo/memo
                           (fn [element_type]
                             (filter-text-attribute
                               (structure element_type))))

   :dropdown-attributes  (core-memo/memo
                           (fn [element_type]
                             (filter-dropdown-attribute
                               (structure element_type))))

   :text-area-attributes (core-memo/memo
                           (fn [element_type]
                             (filter-text-area-attribute
                               (structure element_type))))

   :date-time-attributes (core-memo/memo
                           (fn [element_type]
                             (filter-date-time-attribute
                               (structure element_type))))

   :date-attributes      (core-memo/memo
                           (fn [element_type]
                             (filter-date-attribute
                               (structure element_type))))
   })

(def global-impl
  {:lazy->ElementType (core-memo/ttl
                        (fn [_ type_id]
                          (let [element_type (map->ElementType {:type_id type_id})]
                            (log/trace "lazy->ElementType" "element_type" element_type)
                            (-> element_type
                                (assoc :structure_delay (delay (structure-load element_type)))
                                (with-meta {:impl {:TypeManager (type-manager-impl-factory)}}))))
                        :ttl/threshold 30000)

   })

(extend ElementTypes
  Global
  global-impl)

(def type-manager-impl
  (cache/in-meta-impl
    :TypeManager
    [{:name :structure :args []}
     {:name :text-attributes :args []}
     {:name :file-attributes :args []}
     {:name :dropdown-attributes :args []}
     {:name :text-area-attributes :args []}
     {:name :date-time-attributes :args []}
     {:name :date-attributes :args []}]))

(extend ElementType
  TypeManager
  type-manager-impl)

