(ns com.edocu.users.core
  (:use [com.edocu.users.protocols])
  (:require [com.edocu.users.http :as http]
            [clojure.core.memoize :as memo]
            [clojure.tools.logging :as log]
            [clojure.core.async :as async :refer [go <! timeout alts! promise-chan pipe]]
            [com.edocu.users.configuration :as config])
  (:import [com.edocu.users.protocols
            Users
            User
            OpenUser]))



(defn privilge-impl-factory []
  {:element-types-with-privileges-in-organization (memo/memo
                                                    (fn [this organization privilege]
                                                      (let [result (async/promise-chan)]
                                                        (pipe (http/<element-types-with-privileges-in-organization
                                                                this
                                                                organization
                                                                privilege)
                                                              result)
                                                        result)))

   :privileges-in-element                         (memo/memo
                                                    (fn
                                                      ([this edocu_element]
                                                       (privileges-in-element this edocu_element nil))
                                                      ([this edocu_element attribute_filter?]
                                                       (let [result (async/promise-chan)]
                                                         (pipe (http/<privileges-in-element
                                                                 this
                                                                 edocu_element
                                                                 attribute_filter?)
                                                               result)
                                                         result))))

   :privileges-in-organization                    (memo/memo
                                                    (fn
                                                      ([this organization element_type]
                                                       (privileges-in-organization this organization element_type nil))
                                                      ([this organization element_type attribute_filter?]
                                                       (let [result (async/promise-chan)]
                                                         (pipe (http/<privileges-in-organization
                                                                 this
                                                                 organization
                                                                 element_type
                                                                 attribute_filter?)
                                                               result)
                                                         result))))

   :organizations                                 (memo/memo
                                                    (fn [this]
                                                      (let [result (async/promise-chan)]
                                                        (pipe (http/<organizations
                                                                this)
                                                              result)
                                                        result)))

   :<readable                                     (memo/memo
                                                    (fn [this]
                                                      (let [result (async/promise-chan)]
                                                        (pipe (http/<readable
                                                                this)
                                                              result)
                                                        result)))

   :belongs-to-security-group                     (memo/memo
                                                    (fn
                                                      ([this security_group_id]
                                                       (let [result (async/promise-chan)]
                                                         (pipe (http/<belongs-to-security-group
                                                                 this
                                                                 security_group_id)
                                                               result)
                                                         result))
                                                      ([this organization_id title]
                                                       (let [result (async/promise-chan)]
                                                         (pipe (http/<belongs-to-security-group
                                                                 this
                                                                 organization_id
                                                                 title)
                                                               result)
                                                         result))))})

(def privilege-impl
  {:element-types-with-privileges-in-organization (fn [this organization privilege]
                                                    (let [impl (get-in (meta this) [:impl :Privilege])]
                                                      (apply (:element-types-with-privileges-in-organization impl)
                                                             [this organization privilege])))

   :privileges-in-element                         (fn
                                                    ([this edocu_element]
                                                     (privileges-in-element this edocu_element nil))
                                                    ([this edocu_element attribute_filter?]
                                                     (let [impl (get-in (meta this) [:impl :Privilege])]
                                                       (apply (:privileges-in-element impl)
                                                              [this edocu_element attribute_filter?]))))

   :privileges-in-organization                    (fn
                                                    ([this organization element_type]
                                                     (privileges-in-organization this organization element_type nil))
                                                    ([this organization element_type attribute_filter?]
                                                     (let [impl (get-in (meta this) [:impl :Privilege])]
                                                       (apply (:privileges-in-organization impl)
                                                              [this organization element_type attribute_filter?]))))

   :organizations                                 (fn [this]
                                                    (let [impl (get-in (meta this) [:impl :Privilege])]
                                                      (apply (:organizations impl)
                                                             [this])))

   :<readable                                     (fn [this]
                                                    (let [impl (get-in (meta this) [:impl :Privilege])]
                                                      (apply (:<readable impl)
                                                             [this])))

   :belongs-to-security-group                     (fn
                                                    ([this security_group_id]
                                                     (let [impl (get-in (meta this) [:impl :Privilege])]
                                                       (apply (:belongs-to-security-group impl)
                                                              [this security_group_id])))
                                                    ([this organization_id title]
                                                     (let [impl (get-in (meta this) [:impl :Privilege])]
                                                       (apply (:belongs-to-security-group impl)
                                                              [this organization_id title]))))})

(def director-impl
  {:lazy->User      (fn [_ uid]
                      (let [factory (if (= OPEN_USER_UID uid)
                                      map->OpenUser
                                      map->User)
                            privs_imp (privilge-impl-factory)
                            user (with-meta (factory {:uid uid})
                                            {:impl {:Privilege (if (= OPEN_USER_UID uid)
                                                                 (assoc privs_imp
                                                                   :organizations
                                                                   (fn [_]
                                                                     (go ["Public"])))
                                                                 privs_imp)}})]
                        (log/trace "lazy->User" "uid" uid)
                        user))

   :create->User    (fn [director author uid cn sn mail mobile preferredLanguage password]
                      (go
                        (log/trace "create->User" "uid" uid
                                   "cn" cn
                                   "sn" sn
                                   "mail" mail
                                   "mobile" mobile
                                   "preferredLanguage" preferredLanguage)
                        (let [user (->User uid cn sn mail mobile preferredLanguage password)
                              [created? _] (alts! [(http/<create->User author user) (timeout config/RESPONSE_TIME)])]
                          (when created?
                            (lazy->User director uid)))))

   :<users-metadata (fn [director uids required_info]
                      (log/trace "<users-metadata. uids:" uids "required_info:" required_info)
                      (http/<users-metadata
                        uids
                        required_info))})

(extend Users
  UsersDirector
  director-impl)

(extend User
  Privilege
  privilege-impl)

(extend OpenUser
  Privilege
  privilege-impl)