(ns com.edocu.users.http
  (:require [com.edocu.users.protocols :as users]
            [com.edocu.users.configuration :as config]
            [clojure.tools.logging :as log]
            [cheshire.core :as json]
            [org.httpkit.client :as http]
            [clojure.core.async :refer [go chan <! >! close! put!]]
            [clojure.spec :as s]
            [com.edocu.users.help.spec :as s-help :refer (chan-of)]
            [com.edocu.users.help.sentry :refer [put-in-mdc]]))

(defn- options
  ([user]
   (options user {}))
  ([user others?]
   (merge-with merge
               {:timeout 10000
                :headers {"uid"    (:uid user)
                          "Accept" "application/json"}}
               others?)))

(s/def ::response-channel #(satisfies? clojure.core.async.impl.protocols/ReadPort %))

(s/def ::attribute string?)
(s/def ::type-privilege #{"read" "update" "create"})
(s/def ::attribute-privilege #{"read" "update"})

(s/def ::type (s/coll-of ::type-privilege :kind vector?))
(s/def ::attributes (s/or
                      :with-filter (s/coll-of ::attribute :kind vector?)
                      :without-filter (s/map-of keyword?
                                                (s/coll-of ::attribute-privilege :kind vector?))))

(s/def ::privileges (s/keys :req-un [::type ::attributes]))

(s/def ::privileges-response (s/keys :opt-un [::privileges]))

(defn- construct-user-privileges [body privilege_filter]
  (log/trace "construct-user-privileges" "body:" body "privilege_filter:" privilege_filter)
  (if (s/valid? ::privileges-response body)
    (users/->Privileges
      (if-let [type_privs (get-in body [:privileges :type])]
        (set (map keyword type_privs))
        #{})
      (let [att_privs (if-let [tmp (get-in body [:privileges :attributes])]
                        tmp
                        {})]
        (if (map? att_privs)
          (zipmap
            (keys att_privs)
            (map (fn [privs] (set (map keyword privs))) (vals att_privs)))
          (set (map keyword att_privs)))))
    (do
      (put-in-mdc {"body"             body
                   "privilege_filter" privilege_filter})
      (log/error "Invalid privileges.")
      (users/->Privileges #{} {}))))

(defn <privileges-in-element [user edocu_element privilege_filter]
  (let [base_url (config/USER_PRIVILEGES_IN_ELEMENT (:element_hash edocu_element))
        response_channel (chan-of ::users/privileges)
        default {}]
    (log/trace "privileges-in-element" "user:" user "edocu_element" edocu_element "privilege_filter" privilege_filter "base_url:" base_url)
    (http/get base_url
              (options user {:query-params (if privilege_filter {:only (name privilege_filter)} {})})
              (fn [{:keys [status body error]}]
                (let [response (if (or error (>= status 400))
                                 (do
                                   (if error
                                     (do
                                       (put-in-mdc {"user-info"        (:uid user)
                                                    "edocu_element"    edocu_element
                                                    "privilege_filter" privilege_filter
                                                    "base_url"         base_url
                                                    "status"           status
                                                    "service"          "Nugeta"
                                                    "error"            error})
                                       (log/error "Nugeta error on privileges in element.")))
                                   default)
                                 (try
                                   (json/parse-string body true)
                                   (catch Exception e
                                     (put-in-mdc {"user-info"        (:uid user)
                                                  "edocu_element"    edocu_element
                                                  "privilege_filter" privilege_filter
                                                  "base_url"         base_url
                                                  "status"           status
                                                  "service"          "Nugeta"
                                                  "body"             body})
                                     (log/error e
                                                "Invalid response body from Nugeta on privileges in element.")
                                     default)))]
                  (put! response_channel
                        (construct-user-privileges response privilege_filter)
                        (fn [_] (close! response_channel))))))
    response_channel))

(s/fdef <privileges-in-element
        :args (s/cat :user ::users/user
                     :edocu_element ::users/edocu-element
                     :privilege_filter ::users/type-privilege)
        :ret ::response-channel)

(s/def ::organization-response (s/keys :req-un [::users/organizations]))

(defn <organizations [user]
  (let [response_channel (chan-of ::users/organizations)
        default []
        base_url (config/USER_ORGANIZATIONS)]
    (http/get base_url
              (options user)
              (fn [{:keys [status body error]}]
                (let [response (if (or error (>= status 400))
                                 (do
                                   (if error
                                     (do (put-in-mdc {"user-info" (:uid user)
                                                      "base_url"  base_url
                                                      "status"    status
                                                      "service"   "API old"
                                                      "error"     error})
                                         (log/error error
                                                    "API Old ERROR on user organizations")))
                                   default)
                                 (try
                                   (json/parse-string body true)
                                   (catch Exception e
                                     (put-in-mdc {"user-info" (:uid user)
                                                  "base_url"  base_url
                                                  "status"    status
                                                  "service"   "API old"
                                                  "body"      body})
                                     (log/error e
                                                "Invalid response body from API old on user organizations")
                                     default)))
                      orgs (if (s/valid? ::organization-response response)
                             (:organizations response)
                             (do
                               (put-in-mdc {"user-info" (:uid user)
                                            "base_url"  base_url
                                            "status"    status
                                            "service"   "API old"
                                            "body"      body})
                               (log/error
                                 "Invalid response from API old on user organizations")
                               default))]
                  (put! response_channel orgs (fn [_] (close! response_channel))))))
    response_channel))

(s/fdef <organizations
        :args (s/cat :user ::users/user)
        :ret ::response-channel)

(defn <privileges-in-organization [user organization element_type privilege_filter]
  (let [base_url (config/USER_PRIVILEGES_IN_ORGANIZATION_FOR_ELEMENT_TYPE (:uid organization) element_type)
        response_channel (chan-of ::users/privileges)
        default {}]
    (log/trace "privilege-in-organization:" "user:" user "organization:" organization "element type:" element_type "base_url:" base_url)
    (http/get base_url
              (options user {:query-params (if privilege_filter {:only (name privilege_filter)} {})})
              (fn [{:keys [status body error]}]
                (let [response (if (or error (>= status 400))
                                 (do
                                   (if error
                                     (do
                                       (put-in-mdc {"user-info"        (:uid user)
                                                    "organization"     organization
                                                    "element_type"     element_type
                                                    "privilege_filter" privilege_filter
                                                    "base_url"         base_url
                                                    "status"           status
                                                    "service"          "Nugeta"
                                                    "error"            error})
                                       (log/error "Nugeta ERROR on privileges in organization")))
                                   default)
                                 (try
                                   (json/parse-string body true)
                                   (catch Exception e
                                     (put-in-mdc {"user-info"        (:uid user)
                                                  "organization"     organization
                                                  "element_type"     element_type
                                                  "privilege_filter" privilege_filter
                                                  "base_url"         base_url
                                                  "status"           status
                                                  "service"          "Nugeta"
                                                  "body"             body})
                                     (log/error e
                                                "Invalid response body from Nugeta on privileges in organization")
                                     default)))]
                  (put! response_channel
                        (construct-user-privileges response privilege_filter)
                        (fn [_] (close! response_channel))))))
    response_channel))

(s/fdef <privileges-in-organization
        :args (s/cat :user ::users/user
                     :organization ::users/organization
                     :element_type ::users/id
                     :privilege_filter ::users/type-privilege)
        :ret ::response-channel)

(defn <create->User [author user]
  (let [response_channel (chan-of (s/spec boolean?))]
    (log/trace "create->User" "user:" user)
    (http/post (config/CREATE_USER)
               {:headers {"uid"          (:uid author)
                          "Content-Type" config/USER_CREATE_CONTENT_TYPE
                          "Accept"       config/USER_CREATE_CONTENT_TYPE}
                :body    (json/generate-string user)}
               (fn [{:keys [status body error]}]
                 (let [log_lvl (if error
                                 #'log/error
                                 #'log/trace)]
                   (log_lvl "create->User" "user:" user "status:" status "error:" error "body:" body))
                 (put! response_channel
                       (= 201 status)
                       (fn [_] (close! response_channel)))))
    response_channel))

(s/fdef <create->User
        :args (s/cat :author ::users/user
                     :user ::users/user)
        :ret ::response-channel)

(defn- <ping-belongs-to-security-group [user url]
  (let [response_channel (chan-of (s/spec boolean?))]
    (http/head url
               (options user)
               (fn [{:keys [status error]}]
                 (if error
                   (do
                     (put-in-mdc {"user-info" (:uid user)
                                  "base_url"  url
                                  "status"    status
                                  "service"   "Nugeta"
                                  "error"     error})
                     (log/error "belongs-to-security-group"))
                   (log/trace "belongs-to-security-group" "user:" user "url:" url "status:" status))
                 (put! response_channel
                       (true? (or
                                (= status 200)
                                (= status 204)))
                       (fn [_] (close! response_channel)))))
    response_channel))

(defn <belongs-to-security-group
  ([user security_group_id]
   (log/trace "belongs to security group. user:" user "security_group_id:" security_group_id)
   (<ping-belongs-to-security-group
     user
     (config/CHECK_USER_IN_SECURITY_GROUP
       (:uid user)
       security_group_id)))
  ([user organization_id title]
   (log/trace "belongs to security group. user:" user "organization_id:" organization_id "title:" title)
   (<ping-belongs-to-security-group
     user
     (config/CHECK_USER_IN_SECURITY_GROUP
       (:uid user)
       organization_id
       title))))

(s/fdef <belongs-to-security-group
        :args (s/or
                :by-security-group (s/cat :user ::users/user
                                          :security_group_id ::users/uid)
                :by-organization (s/cat :user ::users/user
                                        :organization_id ::users/uid
                                        :title string?))
        :ret ::response-channel)

(s/def ::element-type-with-privileges-response (s/keys :req-un [::users/element_types]))

(defn <element-types-with-privileges-in-organization [user organization privilege]
  (log/trace "element-types-with-privileges-in-organization:" "user:" user "organization:" organization "privileges:" privilege)
  (let [base_url (config/ELEMENT_TYPES_WITH_PRIVILEGES_IN_ORGANIZATION
                   (:uid user)
                   (:uid organization)
                   (name privilege))
        response_channel (chan-of ::users/element_types)
        default []]
    (http/get base_url
              (options user)
              (fn [{:keys [status body error]}]
                (let [response (if (or error (>= status 400))
                                 (do
                                   (if error
                                     (do
                                       (put-in-mdc {"user-info"        (:uid user)
                                                    "organization"     organization
                                                    "privilege_filter" privilege
                                                    "base_url"         base_url
                                                    "status"           status
                                                    "service"          "Nugeta"
                                                    "error"            error})
                                       (log/error "Nugeta ERROR on element types with privileges in organization")))
                                   default)
                                 (try
                                   (json/parse-string body true)
                                   (catch Exception e
                                     (put-in-mdc {"user-info"        (:uid user)
                                                  "organization"     organization
                                                  "privilege_filter" privilege
                                                  "base_url"         base_url
                                                  "status"           status
                                                  "service"          "Nugeta"
                                                  "body"             body})
                                     (log/error e
                                                "Invalid response body from Nugeta on element types with privileges in organization.")
                                     default)))
                      ets (if (s/valid? ::element-type-with-privileges-response response)
                            (:element_types response)
                            (do
                              (put-in-mdc {"user-info"        (:uid user)
                                           "organization"     organization
                                           "privilege_filter" privilege
                                           "base_url"         base_url
                                           "status"           status
                                           "service"          "Nugeta"
                                           "body"             body})
                              (log/error
                                "Invalid response from Nugeta on element types with privileges in organization.")
                              default))]
                  (put! response_channel ets (fn [_] (close! response_channel))))))
    response_channel))

(s/fdef <element-types-with-privileges-in-organization
        :args (s/cat :user ::users/user
                     :organization ::users/organization
                     :privilege ::users/type-privilege)
        :ret ::response-channel)

(s/def ::users-info (s/map-of keyword? map?))

(defn <users-metadata
  "Return channels with users metadata"
  [uids required_info]
  (let [result (chan-of ::users-info)]
    (http/post (config/USERS_INFO_ABOUT)
               {:headers {"uid"          "edocu-admin"
                          "Content-Type" config/USER_INFO_ABOUT_CONTENT_TYPE
                          "Accept"       config/USER_INFO_ABOUT_CONTENT_TYPE}
                :body    (json/generate-string {:uids uids
                                                :required_info required_info})}
               (fn [{:keys [status body error]}]
                 (if (or error
                         (not= 200 status))
                   (do
                     (close! result)
                     (put-in-mdc {"uids" uids
                                  "required_info" required_info
                                  "status" status})
                     (log/error error "<users-metadata"))
                   (try
                     (let [response (json/parse-string
                                      body
                                      true)]
                       (put! result
                             response
                             (fn [_] (close! result))))
                     (catch Exception e
                       (put-in-mdc "body" body)
                       (log/error error "<users-metadata"))))))
    result))

(s/fdef <users-metadata
        :args (s/cat :uids ::users/uids
                     :required_info ::users/required_info)
        :ret ::s-help/read-channel)

(comment
  (go
    (clojure.pprint/pprint
      (<! (<users-metadata
            #{"Selmeci" "demo1"}
            ["cn" "sn" "mail"])))))