(ns com.edocu.users.core
  (:use [com.edocu.users.protocols])
  (:require [com.edocu.users.http :as http]
            [clojure.core.memoize :as memo]
            [taoensso.timbre :as timbre]
            [clojure.core.async :refer [go chan >! <! <!! timeout alts! thread promise-chan close!]]
            [com.edocu.users.configuration :as config])
  (:import [com.edocu.users.protocols
            Users
            User
            OpenUser]))

(def global-impl
  {:lazy->User   (fn [_ uid]
                   (let [factory (if (= OPEN_USER_UID uid)
                                   map->OpenUser
                                   map->User)
                         user (factory {:uid uid})]
                     (timbre/trace "lazy->User" "uid" uid)
                     user))

   :create->User (fn [_ author uid cn sn mail mobile preferredLanguage password]
                   (thread
                     (timbre/trace "create->User" "uid" uid
                                   "cn" cn
                                   "sn" sn
                                   "mail" mail
                                   "mobile" mobile
                                   "preferredLanguage" preferredLanguage)
                     (let [user (->User uid cn sn mail mobile preferredLanguage password)
                           [created? _] (alts! [(http/create->User author user) (timeout config/RESPONSE_TIME)])]
                       (when created?
                         user))))})

(def privilege-impl
  {:element-types-with-privileges-in-organization (memo/memo
                                                    (fn [this organization privilege]
                                                      (let [result (promise-chan)
                                                            response_chan (chan)]
                                                        (go
                                                          (http/element-types-with-privileges-in-organization
                                                            this
                                                            organization
                                                            privilege
                                                            response_chan)
                                                          (if-let [response (<! response_chan)]
                                                            (>! result response)
                                                            (close! result)))
                                                        result)))

   :privileges-in-element                         (memo/memo
                                                    (fn
                                                      ([this edocu_element]
                                                       (privileges-in-element this edocu_element nil))
                                                      ([this edocu_element attribute_filter?]
                                                       (let [result (promise-chan)
                                                             response_chan (chan)]
                                                         (go
                                                           (http/privileges-in-element
                                                             this
                                                             edocu_element
                                                             attribute_filter?
                                                             response_chan)
                                                           (if-let [response (<! response_chan)]
                                                             (>! result response)
                                                             (close! result)))
                                                         result))))

   :privileges-in-organization                    (memo/memo
                                                    (fn
                                                      ([this organization ^String element_type]
                                                       (privileges-in-organization this organization element_type nil))
                                                      ([this organization ^String element_type attribute_filter?]
                                                       (let [result (promise-chan)
                                                             response_chan (chan)]
                                                         (go
                                                           (http/privileges-in-organization
                                                             this
                                                             organization
                                                             element_type
                                                             attribute_filter?
                                                             response_chan)
                                                           (if-let [response (<! response_chan)]
                                                             (>! result response)
                                                             (close! result)))
                                                         result))))

   :organizations                                 (memo/memo
                                                    (fn [this]
                                                      (let [result (promise-chan)
                                                            response_chan (chan)]
                                                        (go
                                                          (http/organizations
                                                            this
                                                            response_chan)
                                                          (if-let [response (<! response_chan)]
                                                            (>! result response)
                                                            (close! result)))
                                                        result)))

   :belongs-to-security-group                     (memo/memo
                                                    (fn
                                                      ([this ^String security_group_id]
                                                       (let [result (promise-chan)
                                                             response_chan (chan)]
                                                         (go
                                                           (http/belongs-to-security-group
                                                             this
                                                             security_group_id
                                                             response_chan)
                                                           (if-let [response (<! response_chan)]
                                                             (>! result response)
                                                             (close! result)))
                                                         result))
                                                      ([this ^String organization_id ^String title]
                                                       (let [result (promise-chan)
                                                             response_chan (chan)]
                                                         (go
                                                           (http/belongs-to-security-group
                                                             this
                                                             organization_id
                                                             title
                                                             response_chan)
                                                           (if-let [response (<! response_chan)]
                                                             (>! result response)
                                                             (close! result)))
                                                         result))))})

(def open-privilege-impl
  (assoc privilege-impl
    :organizations
    (fn [_]
      (go {:respond (delay ["Public"])}))))

(extend Users
  Global
  global-impl)

(extend User
  Privilege
  privilege-impl)

(extend OpenUser
  Privilege
  open-privilege-impl)
