(ns com.edocu.help.patch
  "Contains help function for JSON PATCH implementation in eDocu"
  ^{:author "roman.selmeci@edocu.com"}
  (:require [clojure.spec.alpha :as s]
            [clojure.core.memoize :as memo])
  (:import [java.util Date]
           [org.apache.commons.validator.routines UrlValidator]))

(def ^:const path-with-value-regex #"/element/\w+/(.+)")
(def ^:private op-with-value #{"add"})

(s/def ::allowed-attribute-types #{"Text" "Dropdown" "Textarea" "DateTime" "Date" "User" "Query" "Element"
                                   "Listing" "SecurityGroup" "Number" "Geo" "Iframe"})
(s/def ::op #{"add" "remove"})
(s/def ::timestamp #(instance? Date %))
(s/def ::value (s/or
                 :uri (s/and string? #(not-empty %)
                             (fn [url]
                               (let [schemes (into-array String ["https" "http"])
                                     ^UrlValidator url-validator (UrlValidator. schemes)]
                                 (.isValid url-validator url))))
                 :string (s/and string? #(not-empty %))
                 :geo (s/coll-of number?)
                 :object map?
                 :number number?
                 :timestamp ::timestamp))
(s/def ::path (s/with-gen
                (s/and string?
                       #(re-matches #"/element/\w+(/.+)?" %))
                #(s/gen #{"/element/ATTRIBUTE"
                          "/element/ATTRIBUTE/-"
                          "/element/ATTRIBUTE/VALUE"})))
(s/def ::text-attribute-path (s/with-gen
                               #(re-matches #"/element/\w+" %)
                               #(s/gen #{"/element/ATTRIBUTE"})))
(s/def ::add-array-attribute-path (s/with-gen
                                    #(re-matches #"/element/\w+/-" %)
                                    #(s/gen #{"/element/ATTRIBUTE/-"})))
(s/def ::remove-array-attribute-path (s/with-gen
                                       #(re-matches path-with-value-regex %)
                                       #(s/gen #{"/element/ATTRIBUTE/VALUE"})))

(defn command-type-dispatch [{:keys [op]}]
  (if (op-with-value op)
    :command-with-value
    :command-common))

(defmulti command-type
          "Return spec for concrete command operation"
          {:arglists '([command])
           :added    "0.1.9"}
          #'command-type-dispatch)

(defmethod command-type :command-with-value [_]
  (s/keys :req-un [::op ::path ::value]))

(defmethod command-type :command-common [_]
  (s/keys :req-un [::op ::path]))

(defmulti command-path
          "Return spec for path in command according to command type"
          {:arglists '([command])
           :added    "0.1.9"}
          #'command-type-dispatch)

(defn valid-add-path?
  "Check if path in add operation is valid"
  [{:keys [path]}]
  (s/valid?
    (s/or
      :text-attribute ::text-attribute-path
      :array-attribute ::add-array-attribute-path)
    path))

(defmethod command-path :command-with-value [_]
  (s/spec valid-add-path?))

(defn valid-remove-path?
  "Check if path in remove operation is valid"
  [{:keys [path]}]
  (s/valid?
    (s/or
      :text-attribute ::text-attribute-path
      :array-attribute ::remove-array-attribute-path)
    path))

(defmethod command-path :command-common [_]
  (s/spec valid-remove-path?))

(s/def ::command (s/and
                   (s/multi-spec command-type #'command-type-dispatch)
                   (s/multi-spec command-path #'command-type-dispatch)))

(s/def ::id string?)
(s/def ::hash string?)
(s/def ::parent (s/keys :req-un [::hash]))
(s/def ::organization (s/keys :req-un [::id]))
(s/def ::operations (s/coll-of ::command :min-count 1))
(s/def ::element-patch (s/keys :req-un [::organization ::operations]
                               :opt-un [::parent]))

(defn is-remove-op?
  "Return true if op is remove operation"
  [op]
  (= "remove" op))

(s/fdef is-remove-op?
        :args (s/cat :op ::op)
        :ret boolean?)

(defn is-add-op?
  "Return true if op is add operation"
  [op]
  (= "add" op))

(s/fdef is-add-op?
        :args (s/cat :op ::op)
        :ret boolean?)

(def path-attribute-regex
  (memo/lru
    (fn [path]
      (keyword
        (second
          (re-matches #"/element/(\w+).*" path))))
    :lru/threshold 50))

(s/def ::attribute keyword?)

(defn path->attribute
  "Extract attribute from path. Return it as keyword"
  [path]
  (path-attribute-regex path))

(s/fdef path->attribute
        :args (s/cat :path ::path)
        :ret ::attribute)

(def path->value-regex
  (memo/lru
    (fn [path]
      (second
        (re-matches path-with-value-regex path)))
    :lru/threshold 50))

(defn path->value
  "Extract value from path"
  [path]
  (path->value-regex path))

(s/fdef path->value
        :args (s/cat :path ::path)
        :ret string?)

(defn operations->attributes
  "Extract distinct attributes used in paths"
  [operations]
  (into #{}
        (map
          (fn [{:keys [path]}] (path->attribute path))
          operations)))

(s/fdef operations->attributes
        :args (s/cat :operations ::operations)
        :ret (s/coll-of keyword? :kind set?))

(defprotocol WithAttributeIn
  "Provide method for fork with atribute data extracted from command and structure."
  (do-with-attribute [_ command attribute attribute_structure]
    "Function for working with extracted attribute and attribute struckture from command"))

(s/def ::required boolean?)
(s/def ::type string?)
(s/def ::order number?)
(s/def ::key string?)
(s/def ::modifiable boolean?)
(s/def ::values (s/coll-of string?))
(s/def ::markdown boolean?)
(s/def ::multiple boolean?)
(s/def ::query string?)
(s/def ::attribute (s/keys :req-un [::required ::type ::order ::key]
                           :opt-un [::values ::markdown ::multiple ::query ::modifiable]))
(s/def ::attributes (s/map-of keyword? ::attribute))

(s/fdef do-with-attribute
        :args (s/cat :this #(satisfies? WithAttributeIn %)
                     :command ::command
                     :attribute ::attribute
                     :attribute_structure ::attribute)
        :ret any?)

(defn with-attribute-in
  "Wrap extracting of attribute from path and structure.
  Send it to body function [coomand attribute attribute_structure] and return value from it."
  [structure {:keys [path] :as command} handler]
  (let [attribute (path->attribute path)
        attribute_structture (attribute structure)]
    (do-with-attribute handler command attribute attribute_structture)))

(s/fdef with-attribute-in
        :args (s/cat :structure ::attributes
                     :command ::command
                     :handler #(satisfies? WithAttributeIn %)))