(ns com.edocu.configuration.core
  (:use com.edocu.configuration.constants)
  (:require [taoensso.timbre :as timbre]
            [com.edocu.configuration.etcd :as etcd]
            [clojurewerkz.urly.core :as urly]
            [cheshire.core :as json]
            [environ.core :as e]))

(def ENV (atom nil))

(defn env []
  (if-not @ENV
    (reset! ENV
            (or
              (e/env :env)
              :dev)))
  @ENV)

;############################# Defaults ################################

(def ^:const REDIS_CONFIG_DEFAULT {:pool {}
                                   :spec {:host (or
                                                  (e/env :redis)
                                                  REDIS_HOST)
                                          :port 6379
                                          :db   8}})

(def ^:const ^String MONGODB_REPLICAS_URI_DEFAULT (case (env)
                                                    ("prod" "prod_unsecure") [{:uri "mongodb.1.db.prod.edocu.local" :port 27017}
                                                                              {:uri "mongodb.2.db.prod.edocu.local" :port 27017}
                                                                              {:uri "mongodb.3.db.prod.edocu.local" :port 27017}]
                                                    [{:uri "mongodb1.db.dev.edocu.local" :port 27017}
                                                     {:uri "mongodb2.db.dev.edocu.local" :port 27017}
                                                     {:uri "mongodb3.db.dev.edocu.local" :port 27017}]))
(def ^:const ^String MONGODB_DB_DEFAULT (case (env)
                                          ("prod" "prod_unsecure") "eDocu-production"
                                          "eDocu-development"))


(defn with-default
  ([path default]
   (with-default
     path
     default
     identity))
  ([path default data_parser]
   (if-let [data (etcd/get-config-for
                   path)]
     (data_parser data)
     default)))

;############################ Base URL ###############################

(def base-url-atom (atom nil))

(defn base-url []
  (when-not @base-url-atom
    (reset! base-url-atom
            (with-default
              BASE_URL_PATH
              BASE_URL_DEFAULT))
    (timbre/trace "base-url" @base-url-atom))
  @base-url-atom)

;######################################################################

;############################## SMTP ##################################

(def smtp-atom (atom nil))

(defn- parse-user-authentification [user_info]
  (if user_info
    (let [[user pass] (clojure.string/split user_info #":")
          result {}
          result (if user (assoc result :user user) result)
          result (if pass (assoc result :pass pass) result)]
      result)
    {}))

(defn- parse-protocol [protocol]
  (case protocol
    "smtps" {:ssl :y}
    "smtpd" {:tls :y}
    {}))

(defn- parse-smtp-config [smtp_url]
  (let [urly_map (urly/as-map (urly/url-like smtp_url))]
    (merge urly_map
           (parse-user-authentification (:user-info urly_map))
           (parse-protocol (:protocol urly_map)))))

(defn smtp []
  (when-not @smtp-atom
    (reset! smtp-atom
            (parse-smtp-config
              (with-default
                SMTP_PATH
                SMTP_DEFAULT)))
    (timbre/trace "smtp" @smtp-atom))
  @smtp-atom)

;######################################################################

;##################### Notifications Email ############################

(def notifications-email-atom (atom nil))

(defn notifications-email []
  (when-not @notifications-email-atom
    (reset! notifications-email-atom
            (with-default
              NOTIFICATION_EMAIL_PATH
              NOTIFICATION_EMAIL_DEFAULT))
    (timbre/trace "notifications-email" @notifications-email-atom))
  @notifications-email-atom)

;######################################################################

;###################### Zookeepers Servers ############################

(def zookeepers-servers-atom (atom nil))

(defn zookeepers-servers []
  (when-not @zookeepers-servers-atom
    (reset! zookeepers-servers-atom
            (with-default
              ZOOKEPER_SERVERS_PATH
              ZOOKEPER_SERVERS_DEFAULT))
    (timbre/trace "zookeepers-servers" @zookeepers-servers-atom))
  @zookeepers-servers-atom)

;######################################################################

;######################## Kafka Brokers ###############################

(def kafka-zookeeper-atom (atom nil))

(defn kafka-zookeeper []
  (when-not @kafka-zookeeper-atom
    (reset! kafka-zookeeper-atom
            (with-default
              KAFKA_ZOOKEEPER_PATH
              KAFKA_ZOOKEEPER_DEFAULT
              #(json/parse-string % true)))
    (timbre/trace "kafka-zookeeper" @kafka-zookeeper-atom))
  @kafka-zookeeper-atom)

;######################################################################

;######################### Redis Config ###############################

(def redis-config-atom (atom nil))

(defn redis-config []
  (when-not @redis-config-atom
    (reset! redis-config-atom
            (with-default
              REDIS_CONFIG_PATH
              REDIS_CONFIG_DEFAULT
              #(json/parse-string % true)))
    (timbre/trace "redis-config" @redis-config-atom))
  @redis-config-atom)

;######################################################################

;############################ CBR topis ###############################

(def cbr-topic-atom (atom nil))

(defn cbr-topic []
  (when-not @cbr-topic-atom
    (reset! cbr-topic-atom
            (with-default
              CBR_TOPIC_PATH
              CBR_TOPIC_DEFAULT))
    (timbre/trace "cbr-topic" @cbr-topic-atom))
  @cbr-topic-atom)

;######################################################################

;########################## Register topis ############################

(def register-topic-atom (atom nil))

(defn register-topic []
  (when-not @register-topic-atom
    (reset! register-topic-atom
            (with-default
              REGISTER_TOPIC_PATH
              REGISTER_TOPIC_DEFAULT))
    (timbre/trace "register-topic" @register-topic-atom))
  @register-topic-atom)

;######################################################################

;########################## Register topis ############################

(def service-error-topic-atom (atom nil))

(defn service-error-topic []
  (when-not @service-error-topic-atom
    (reset! service-error-topic-atom
            (with-default
              SERVICE_ERROR_TOPIC_PATH
              SERVICE_ERROR_TOPIC_DEFAULT))
    (timbre/trace "service-error-topic" @service-error-topic-atom))
  @service-error-topic-atom)

;######################################################################

;########################## Register topis ############################

(def malformed-message-topic-atom (atom nil))

(defn malformed-message-topic []
  (when-not @malformed-message-topic-atom
    (reset! malformed-message-topic-atom
            (with-default
              MALFORMED_MESSAGE_TOPIC_PATH
              MALFORMED_MESSAGE_TOPIC_DEFAULT))
    (timbre/trace "malformed-message-topic" @malformed-message-topic-atom))
  @malformed-message-topic-atom)

;######################################################################

;##################### MongoDB Replicas URI ###########################

(def mongodb-replicas-uri-atom (atom nil))

(defn mongodb-replicas-uri []
  (when-not @mongodb-replicas-uri-atom
    (reset! mongodb-replicas-uri-atom
            (with-default
              MONGODB_REPLICAS_URI_PATH
              MONGODB_REPLICAS_URI_DEFAULT
              #(json/parse-string % true)))
    (timbre/trace "mongodb-replicas-uri" @mongodb-replicas-uri-atom))
  @mongodb-replicas-uri-atom)

;######################################################################

;############################ MongoDB DB ##############################

(def mongodb-db-atom (atom nil))

(defn mongodb-db []
  (when-not @mongodb-db-atom
    (reset! mongodb-db-atom
            (with-default
              MONGODB_DB_PATH
              MONGODB_DB_DEFAULT))
    (timbre/trace "mongodb-db" @mongodb-db-atom))
  @mongodb-db-atom)

;######################################################################

;########################### Invegration ##############################

(def integration-env-atom (atom nil))

(defn integration-env []
  (when-not @integration-env-atom
    (reset! integration-env-atom
            (with-default
              INTEGRATION_ENV_PATH
              INTEGRATION_ENV))
    (timbre/trace "integration-env" @integration-env-atom))
  @integration-env-atom)

;######################################################################
