(ns com.edocu.configuration.core
  (:require [com.edocu.configuration.etcd :as etcd]
            [taoensso.timbre :as timbre]
            [clojurewerkz.urly.core :as urly]
            [cheshire.core :as json]))

(def ^:const ^String BASE_URL_PATH "base-url")
(def ^:const ^String EMAIL_PREFIX "email/")
(def ^:const ^String SMTP_PATH (str EMAIL_PREFIX "smtp"))
(def ^:const ^String NOTIFICATION_EMAIL_PATH (str EMAIL_PREFIX "addresses/notifications"))
(def ^:const ^String KAFKA_PATH "kafka/")
(def ^:const ^String KAFKA_ZOOKEEPER_PATH (str KAFKA_PATH "zookeper"))
(def ^:const ^String REDIS_CONFIG_PATH "redis/config")
(def ^:const ^String ZOOKEPER_SERVERS_PATH "zookeeper/servers")

(def ^:const ^String REDIS_HOST "redis.db.dev.edocu.local")
(def ^:const ^String BASE_URL_DEFAULT "https://edocu.service.dev.edocu.local")
(def ^:const ^String SMTP_DEFAULT "")
(def ^:const ^String NOTIFICATION_EMAIL_DEFAULT "notifications@edocu.eu")
(def ^:const ^String ZOOKEPER_SERVERS_DEFAULT "kafka.mq.dev.edocu.local:2181")
(def ^:const KAFKA_ZOOKEEPER_DEFAULT (json/generate-string {"zookeeper.connect" ZOOKEPER_SERVERS_DEFAULT}))
(def ^:const REDIS_CONFIG_DEFAULT (json/generate-string
                                    {:pool {}
                                     :spec {:host REDIS_HOST
                                            :port 6379
                                            :db   10}}))

;############################ Base URL ###############################

(def base-url-promise (promise))

(defn base-url []
  (if-not (realized? base-url-promise)
    (do
      (deliver base-url-promise
               (etcd/get-config-for
                 BASE_URL_PATH
                 BASE_URL_DEFAULT))
      (timbre/trace "base-url" @base-url-promise)))
  @base-url-promise)

;######################################################################

;############################## SMTP ##################################

(def smtp-promise (promise))

(defn- parse-user-authentification [user_info]
  (if user_info
    (let [[user pass] (clojure.string/split user_info #":")
          result {}
          result (if user (assoc result :user user) result)
          result (if pass (assoc result :pass pass) result)]
      result)
    {}))

(defn- parse-protocol [protocol]
  (case protocol
    "smtps" {:ssl :y}
    "smtpd" {:tls :y}
    {}))

(defn- parse-smtp-config [smtp_url]
  (let [urly_map (urly/as-map (urly/url-like smtp_url))]
    (merge urly_map
           (parse-user-authentification (:user-info urly_map))
           (parse-protocol (:protocol urly_map)))))

(defn smtp []
  (if-not (realized? smtp-promise)
    (do
      (deliver smtp-promise
               (parse-smtp-config
                 (etcd/get-config-for
                   SMTP_PATH
                   SMTP_DEFAULT)))
      (timbre/trace "smtp" @smtp-promise)))
  @smtp-promise)

;######################################################################

;##################### Notifications Email ############################

(def notifications-email-promise (promise))

(defn notifications-email []
  (if-not (realized? notifications-email-promise)
    (do
      (deliver notifications-email-promise
               (etcd/get-config-for
                 NOTIFICATION_EMAIL_PATH
                 NOTIFICATION_EMAIL_DEFAULT))
      (timbre/trace "notifications-email" @notifications-email-promise)))
  @notifications-email-promise)

;######################################################################

;###################### Zookeepers Servers ############################

(def zookeepers-servers-promise (promise))

(defn zookeepers-servers []
  (if-not (realized? zookeepers-servers-promise)
    (do
      (deliver zookeepers-servers-promise
               (etcd/get-config-for
                 ZOOKEPER_SERVERS_PATH
                 ZOOKEPER_SERVERS_DEFAULT))
      (timbre/trace "zookeepers-servers" @zookeepers-servers-promise)))
  @zookeepers-servers-promise)

;######################################################################

;######################## Kafka Brokers ###############################

(def kafka-zookeeper-promise (promise))

(defn kafka-zookeeper []
  (when-not (realized? kafka-zookeeper-promise)
    (deliver kafka-zookeeper-promise
             (json/parse-string
               (etcd/get-config-for
                 KAFKA_ZOOKEEPER_PATH
                 KAFKA_ZOOKEEPER_DEFAULT)
               false))
    (timbre/trace "kafka-zookeeper" @kafka-zookeeper-promise))
  @kafka-zookeeper-promise)

;######################################################################

;######################### Redis Config ###############################

(def redis-config-promise (promise))

(defn redis-config []
  (when-not (realized? redis-config-promise)
    (deliver redis-config-promise
             (json/parse-string
               (etcd/get-config-for
                 REDIS_CONFIG_PATH
                 REDIS_CONFIG_DEFAULT)
               true))
    (timbre/trace "redis-config-promise" @redis-config-promise))
  @redis-config-promise)

;######################################################################