(ns com.edocu.communication.core
  (:require
    #?(:clj
        [clj.com.edocu.communication.kafka.core :as clj-kafka]
       :cljs
       [cljs.com.edocu.communication.kafka.core :as cljs-kafka])
    #?(:clj
        [clojure.tools.logging :as log]
       :cljs
       [taoensso.timbre :as log])
    #?(:clj
        [clojure.core.async :refer [chan go go-loop <! >!]]
       :cljs
       [cljs.core.async :refer [chan <! >!]])
    #?(:clj
        [environ.core :refer [env]])
    #?(:clj
        [clojure.core.memoize :as memo])
    #?(:clj
        [com.edocu.help.sentry :as sentry])
    #?(:cljs [cljs.nodejs :as nodejs])
    #?(:clj
        [clojure.spec.alpha :as s]
       :cljs [cljs.spec :as s])
        [com.edocu.communication.protocols :as prot])
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go go-loop]])))

(defrecord Communicator [stop_check kf]
  prot/IMessageManagement
  (send-message! [_ topic message]
    (prot/send-message!
      kf
      topic
      message))

  (deliver-message! [_ topic original-topic message]
    (prot/deliver-message!
      kf
      topic
      original-topic
      message))

  prot/ITopicManagement
  (register-topics! [this topics]
    (prot/register-topics!
      kf
      topics))

  (subscribe-to-topic [_ topic]
    (prot/subscribe-to-topic
      kf
      (prot/->kafka-topic topic)))

  prot/IErrorsManagement
  (send-malformed-message-report! [_ message]
    (prot/send-malformed-message-report!
      kf
      message))

  (send-service-error-report! [_ message]
    (prot/send-malformed-message-report!
      kf
      message)))

(def stop-check #?(:clj  (memo/ttl
                           (fn []
                             (= "true" (env :stop)))
                           :ttl/threshold 15000)
                   :cljs (fn [] false)))

(defrecord Director []
  prot/ICommunicationFactory
  (create->Communicator [_ consumer_config]
    (->Communicator
      stop-check
      (#?(:clj  clj-kafka/->Communicator
          :cljs cljs-kafka/->Communicator)
        stop-check
        consumer_config))))

(defn report-service-error! [report]
  (let [comm (prot/create->Communicator
               (->Director)
               {:group.id "service-error-report"})]
    (prot/send-service-error-report! comm report)))

(defn report-malformed-message! [report]
  (let [comm (prot/create->Communicator
               (->Director)
               {:group.id "malformed-message-report"})]
    (prot/send-malformed-message-report! comm report)))

(defn process-message-on
  ([group_id topic callback]
   (process-message-on
     prot/default-topic-parser
     prot/->EventMessage
     group_id topic callback))
  ([topic-parser construct-message group_id topic callback]
   (let [comm (prot/create->Communicator
                (->Director)
                {:group.id group_id})]
     (prot/register-topics! comm [topic])
     (let [c (prot/subscribe-to-topic comm
                                      topic)]
       (go-loop [msg (<! c)]
         (when msg
           (log/debug "process-message-on:" "topic:" topic "group_id:" group_id "received message:" msg)
           (try
             (let [msg_topic (:key msg)]
               (callback (construct-message
                           msg_topic
                           (topic-parser msg_topic)
                           (:value msg))))
             #?(:clj  (catch Exception e
                        (sentry/put-in-mdc {:topic    topic
                                            :group_id group_id
                                            :msg      msg})
                        (log/error e "process-message-on"))
                :cljs (catch :default e
                        (log/error e "process-message-on"))))
           (if-not ((:stop_check comm))
             (recur (<! c))
             (log/trace "unsubscribe from topic:" topic)))))
     comm)))

(defn send-message!
  "Send message to message broker"
  [topic message]
  (let [comm (prot/create->Communicator
               (->Director)
               {:group.id "eDocu-Sender"})]
    (prot/send-message!
      comm
      topic
      message)))

(s/fdef send-message!
        :args (s/cat :topic ::prot/topic
                     :message map?))

#?(:cljs (nodejs/enable-util-print!))
#?(:cljs (def -main (fn []
                      (taoensso.timbre/set-level! :trace)
                      nil)))
#?(:cljs (set! *main-cli-fn* -main))

(comment
  (let [topic "cljs.edocu.communication.test.test2"]
    (process-message-on
      "cljs-communication"
      topic
      (fn [message]
        #?(:clj  (clojure.pprint/pprint message)
           :cljs (cljs.pprint/pprint message))))
    #_(send-message!
        topic
        {:msg "Ahoj7"})))