(ns com.edocu.communication.core
  (:use com.edocu.communication.protocols)
  (:require [com.edocu.communication.kafka.core :as kafka]
            [com.edocu.configuration.core :as edocu-config]
            [clojure.tools.logging :as log]
            [clojure.core.async :refer [chan go go-loop <! >!]]
            [environ.core :refer [env]]
            [clojure.core.memoize :as memo]
            [com.edocu.help.sentry :as sentry]))

(defrecord Director [])

(defrecord Communicator [stop_check kf]
  IMessageManagement

  (send-message! [_ topic message]
    (send-message!
      kf
      topic
      message))

  ITopicManagement
  (register-topics! [_ topics]
    (register-topics!
      kf
      topics))

  (subscribe-to-topic [_ topic callback_chan]
    (subscribe-to-topic
      kf
      (->kafka-topic topic)
      callback_chan))

  IErrorsManagement
  (send-malformed-message-report! [_ message]
    (send-malformed-message-report!
      kf
      message))

  (send-service-error-report! [_ message]
    (send-malformed-message-report!
      kf
      message)))

(def stop-check (memo/ttl
                  (fn []
                    (= "true" (env :stop)))
                  :ttl/threshold 15000))

(extend Director
  ICommunicationFactory
  {:create->Communicator (fn [_ consumer_config]
                           (->Communicator
                             stop-check
                             (kafka/->Communicator
                               stop-check
                               consumer_config)))})

(defn report-service-error! [report]
  (let [comm (create->Communicator
               (->Director)
               {:group.id "service-error-report"})]
    (send-service-error-report! comm report)))

(defn report-malformed-message! [report]
  (let [comm (create->Communicator
               (->Director)
               {:group.id "malformed-message-report"})]
    (send-malformed-message-report! comm report)))

(defn process-message-on
  ([group_id topic callback]
   (process-message-on
     default-topic-parser
     ->EventMessage
     group_id topic callback))
  ([topic-parser construct-message group_id topic callback]
   (let [comm (create->Communicator
                (->Director)
                {:group.id group_id})
         c (chan 1024)]
     (register-topics! comm topic)
     (go-loop [msg (<! c)]
       (when msg
         (log/debug "process-message-on:" "topic:" topic "group_id:" group_id "received message:" msg)
         (try
           (let [msg_topic (:key msg)]
             (callback (construct-message msg_topic (topic-parser msg_topic) (:value msg))))
           (catch Exception e
             (sentry/put-in-mdc {:topic    topic
                                 :group_id group_id
                                 :msg      msg})
             (log/error e "process-message-on")))
         (if-not ((:stop_check comm))
           (recur (<! c))
           (log/trace "unsubscribe from topic:" topic))))
     (subscribe-to-topic comm
                         topic
                         c)
     comm)))
