(ns com.edocu.communication.kafka.core
  (:use com.edocu.communication.protocols)
  (:require [taoensso.timbre :as timbre]
            [com.edocu.configuration.core :as config]
            [clojure.core.async :refer [>!!]]
            [clj-kafka.new.producer :as kafka-producer]
            [clj-kafka.zk :as zk]
            [clj-kafka.consumer.zk :as kafka-consumer]
            [clj-kafka.core :as kafka-core]))

(def producer-atom (atom nil))

(defrecord Communicator [consumer_config])

(def communication-factory-impl
  {:create->Communicator (fn [_ consumer_config]
                           (when-not @producer-atom
                             (reset!
                               producer-atom
                               (kafka-producer/producer
                                 {"bootstrap.servers" (zk/broker-list
                                                        (zk/brokers
                                                          (config/kafka-zookeeper)))}
                                 (kafka-producer/byte-array-serializer)
                                 (kafka-producer/byte-array-serializer))))
                           (->Communicator
                             (zipmap
                               (map name (keys consumer_config))
                               (vals consumer_config))))})

(def message-management-impl
  {:route-message! (fn [_ topic message]
                     (timbre/debug "route-message!:" "topic:" topic "message:" message)
                     (kafka-producer/send
                       @producer-atom
                       (kafka-producer/record
                         topic
                         (.getBytes (:topic message))
                         (message-body message))))

   :send-message!  (fn [communicator topic message]
                     (timbre/debug "send-message!:" "topic:" topic "message:" message)
                     (route-message!
                       communicator
                       (config/cbr-topic)
                       (create->SendMessage
                         topic
                         message)))})

(def topic-management-impl
  {:register-topics!   (fn [communicator topics]
                         (timbre/debug ":register-topic!" "topics:" topics)
                         (send-message!
                           communicator
                           (config/register-topic)
                           {:topic topics}))

   :subscribe-to-topic (fn [comminicator topic callback_chan]
                         (future
                           (kafka-core/with-resource
                             [c (kafka-consumer/consumer
                                  (merge
                                    (config/kafka-zookeeper)
                                    (:consumer_config comminicator)))]
                             kafka-consumer/shutdown
                             (doseq [{:keys [key value]} (kafka-consumer/messages c topic)]
                               (>!! callback_chan
                                    (create->ReceiveMessage
                                      (String. key)
                                      value))))))})

(def error-management-impl
  {:send-malformed-message-report! (fn [communicator message]
                                     (send-message!
                                       communicator
                                       (config/malformed-message-topic)
                                       message))

   :send-service-error-report!     (fn [communicator message]
                                     (send-message!
                                       communicator
                                       (config/service-error-topic)
                                       message))})