(ns com.edocu.communication.core
  (:use com.edocu.communication.protocols)
  (:require [com.edocu.communication.kafka.core :as comm-impl]
            [taoensso.timbre :as timbre]
            [clojure.core.async :refer [chan go go-loop <! >! buffer]])
  (:import [com.edocu.communication.kafka.core Communicator]))

(defrecord Director [])

(extend Director
  ICommunicationFactory
  comm-impl/communication-factory-impl)

(extend Communicator
  IMessageManagement
  comm-impl/message-management-impl

  ITopicManagement
  comm-impl/topic-management-impl

  IErrorsManagement
  comm-impl/error-management-impl)

(defmacro def-process-message-on []
  `(defn ~'process-message-on [~'topic ~'callback]
     (let [group_id# (str ~'topic "---" (name ~'callback))
           comm# (create->Communicator
                   (->Director)
                   {:group.id group_id#})
           c# (chan (buffer 1024))]
       (register-topics! comm# ~'topic)
       (go-loop [msg# (<! c#)]
                (when msg#
                  (timbre/debug "process-message-on:" "topic:" ~'topic "group_id:" group_id# "received message:" msg#)
                  (~'callback (:topic msg#) (parsed-message-body msg#))
                  (recur (<! c#))))
       (subscribe-to-topic comm#
                           ~'topic
                           c#))))

(defmacro def-subcribe-process [doc process_name service & yield]
  `(defn ~(symbol process_name) ~doc [~'topic ~'body]
     (go
       (try
         (timbre/trace ~process_name "topic:" ~'topic "body:" ~'body)
         (let [~'message (construct-message ~'topic ~'body)]
           ~@yield)
         (catch Exception e#
           (timbre/error ~process_name "service:" ~service "topic:" ~'topic "body:" ~'body "error:" e#))))))

(defn report-service-error! [report]
  (let [comm (create->Communicator
               (->Director)
               {:group.id "service-error-report"})]
    (send-service-error-report! comm report)))

(defn report-malformed-message! [report]
  (let [comm (create->Communicator
               (->Director)
               {:group.id "malformed-message-report"})]
    (send-malformed-message-report! comm report)))
