(ns com.edocu.communication.core
  (:use [com.edocu.communication.protocols]))

(require '[com.edocu.communication.kafka.core :as comm-impl]
         '[com.edocu.configuration.core :as config]
         '[taoensso.timbre :as timbre]
         '[clojure.core.async :refer [<! go-loop go chan buffer]])

#?(:clj (require '[clj-kafka.zk :as kafka-zk]
                 '[clj-kafka.new.producer :as kafka-producer]))


(defrecord Director []
  ICommunicationFactory
  (create->Communicator [_ consumer_config]
    #?@(:clj ((when-not (realized? comm-impl/producer-promise)
                (deliver
                  comm-impl/producer-promise
                  (kafka-producer/producer
                    {"bootstrap.servers" (kafka-zk/broker-list
                                           (kafka-zk/brokers
                                             (config/kafka-zookeeper)))}
                    (kafka-producer/byte-array-serializer)
                    (kafka-producer/byte-array-serializer))))
               (comm-impl/->Communicator
                 (zipmap
                   (map name (keys consumer_config))
                   (vals consumer_config))))

        :default ((timbre/error "create->Communicator: Not implemented")))))

(defrecord ElementType [_id]
  #?@(:clj (Object
             (toString [this] (pr-str this)))))

(defrecord Organization [_id]
  #?@(:clj (Object
             (toString [this] (pr-str this)))))

(defrecord Action [_id]
  #?@(:clj (Object
             (toString [this] (pr-str this)))))

(defrecord User [_id]
  #?@(:clj (Object
             (toString [this] (pr-str this)))))

(defrecord MessageHeaders [^ElementType element_type
                           ^Organization organization
                           ^Action action
                           ^User user]
  #?@(:clj (Object
             (toString [this] (pr-str this)))))

(defrecord EventMessage [^MessageHeaders headers
                         body]
  #?@(:clj (Object
             (toString [this] (pr-str this)))))

(defrecord MalformedEventMessageReport [^String original_topic
                                        ^EventMessage original_message
                                        ^String reporting_service
                                        description]
  #?@(:clj (Object
             (toString [this] (pr-str this)))))

(defrecord ServiceErrorMessageReport [^String service
                                      ^EventMessage original_message
                                      ^String reporting_service
                                      request
                                      respond]
  #?@(:clj (Object
             (toString [this] (pr-str this)))))

(defn construct-message [topic body]
  (let [[_ element_type organization action user] (clojure.string/split topic #"\.")
        result (->EventMessage
                 (->MessageHeaders
                   (->ElementType element_type)
                   (->Organization organization)
                   (->Action action)
                   (->User user))
                 body)]
    (timbre/trace "construct-message:" result)
    result))

(defn report-service-error! [report]
  (let [comm (create->Communicator
               (->Director)
               {:group.id "business-rules-service-error-report"})]
    (send-service-error-report! comm report)))

(defn report-malformed-message! [report]
  (let [comm (create->Communicator
               (->Director)
               {:group.id "business-rules-malformed-message-report"})]
    (send-malformed-message-report! comm report)))

(defmacro process-message-on [topic callback]
  `(let [group_id# ~(str topic "---" (name callback))
         comm# (create->Communicator
                 (->Director)
                 {:group.id group_id#})
         c# (chan (buffer 1024))]
     (register-topics! comm# ~topic)
     (go-loop [msg# (<! c#)]
              (when msg#
                (timbre/debug "process-message-on:" "topic:" ~topic "group_id:" group_id# "received message:" msg#)
                (~callback (:topic msg#) (parsed-message-body msg#))
                (recur (<! c#))))
     (subscribe-to-topic comm#
                         ~topic
                         c#)))

(defmacro def-subcribe-process [doc process_name service & yield]
  `(defn ~(symbol process_name) ~doc [~'topic ~'body]
     (go
       (try
         (timbre/trace ~process_name "topic:" ~'topic "body:" ~'body)
         (let [~'message (construct-message ~'topic ~'body)]
           ~@yield)
         (catch Exception e#
           (timbre/error ~process_name "topic:" ~'topic "mailing:" ~'body "error:" e#)
           (report-malformed-message! (->MalformedEventMessageReport
                                        ~'topic
                                        (construct-message ~'topic ~'body)
                                        ~service
                                        e#)))))))


